use ui;
use layout;
use core:geometry;
use core:lang;
use lang:bs:macro;

class TreeWindow extends Frame {
	private TreePainter painter;
	private Map<Key, fn()->void> keymap;
	private Bool mouseDown;

	init() {
		init("Treez", Size(500, 500)) {
			painter();
		}

		painter(painter);
		create();

		keymap.put(Key:i, &this.insert());
		keymap.put(Key:d, &this.delete());
		keymap.put(Key:c, &this.clear());
		keymap.put(Key:l, &this.setLeft());
		keymap.put(Key:r, &this.setRight());
		keymap.put(Key:j, &this.setLeft());
		keymap.put(Key:k, &this.setRight());
		keymap.put(Key:esc, &this.abort());
		keymap.put(Key:e, &this.loadExample());
		keymap.put(Key:q, &this.clearAll());
		keymap.put(Key(60), &this.insertPipe()); // < on swedish keyboards
		keymap.put(Key:z, &this.insertPipe());
		keymap.put(Key:a, &this.layoutNode());

		var me = this;
		keymap.put(Key:num0, () => me.numKey(0));
		keymap.put(Key:num1, () => me.numKey(1));
		keymap.put(Key:num2, () => me.numKey(2));
		keymap.put(Key:num3, () => me.numKey(3));
		keymap.put(Key:num4, () => me.numKey(4));
		keymap.put(Key:num5, () => me.numKey(5));
		keymap.put(Key:num6, () => me.numKey(6));
		keymap.put(Key:num7, () => me.numKey(7));
		keymap.put(Key:num8, () => me.numKey(8));
		keymap.put(Key:num9, () => me.numKey(9));
	}

	Bool onClick(Bool down, Point pt, MouseButton button) : override {
		mouseDown = down;
		painter.mouseClicked(down, pt, button);
		repaint();
		true;
	}

	Bool onMouseMove(Point pt) : override {
		lastMouse = pt;
		if (painter.mouseMoved(pt, mouseDown)) {
			repaint();
		}
		true;
	}

	void onMouseLeave() : override {
		mouseDown = false;
	}

	Bool onKey(Bool pressed, Key key, Modifiers modifiers) : override {
		if (!pressed)
			return false;

		if (modifiers == Modifiers:none) {
			if (keymap.has(key)) {
				keymap.get(key).call();
			}
			true;
		} else if (modifiers == Modifiers:shift) {
			if (key == Key:l | key == Key:j) {
				painter.rotateLeft(lastMouse);
				repaint();
			} else if (key == Key:r | key == Key:k) {
				painter.rotateRight(lastMouse);
				repaint();
			} else if (key == Key:c) {
				painter.cloneTree(lastMouse);
				repaint();
			} else if (key == Key:d) {
				painter.deleteTree(lastMouse);
				repaint();
			}
		} else if (modifiers == Modifiers:ctrl) {
			if (key == Key:z) {
				painter.undo();
				repaint();
			}
		}

		false;
	}

	private Point lastMouse;

	private Nat number;

	private void insert() {
		painter.insert(number, lastMouse);
		number = 0;
		repaint();
	}

	private void insertPipe() {
		painter.insertPipe(lastMouse);
		repaint();
	}

	private void delete() {
		painter.delete(lastMouse);
		repaint();
	}

	private void clear() {
		painter.clear(lastMouse);
		repaint();
	}

	private void clearAll() {
		painter.clear();
		repaint();
	}

	private void abort() {
		number = 0;
		painter.abort();
		repaint();
	}

	private void setLeft() {
		painter.startDrag(lastMouse, DragFrom:left);
	}

	private void setRight() {
		painter.startDrag(lastMouse, DragFrom:right);
	}

	private void numKey(Nat num) {
		number = number * 10 + num;
	}

	private void layoutNode() {
		painter.layoutNode(lastMouse);
		repaint();
	}

	private void loadExample() {
		Package here = named{};
		if (url = here.url) {
			url = url / "${number}.tree";
			try {
				if (!url.exists()) {
					showMessage(this, "File not found", "The file ${url} does not exist.");
				} else if (node = parseFile(url)) {
					painter.load(node);
					repaint();
				}
			} catch (Exception e) {
				showMessage(this, "File format error", "Failed to parse the file: ${e}");
			}
		}
		number = 0;
	}
}

enum DragFrom {
	left, right
}

class TreePainter extends Painter {
	init() {
		init {}
	}

	Point offset;

	Node[] nodes;

	// Last clicked.
	Node? clicked;

	// Mouse origin when dragging.
	Point mouseOrigin;

	// Dragging "from".
	DragFrom edgeFrom;
	Node? from;

	// Undo stack. We cap this at ~10-15 operations.
	Node[][] undoStack;

	Bool render(Size me, Graphics g) {
		g.transform(translate(-offset));
		g.lineWidth = 1.5;

		for (node in nodes) {
			node.draw(g);
		}

		if (from) {
			if (edgeFrom == DragFrom:left) {
				g.line(from.bottomLeft, mouseOrigin, nodeBrush);
			} else {
				g.line(from.bottomRight, mouseOrigin, nodeBrush);
			}
		}

		false;
	}

	void mouseClicked(Bool down, Point pt, MouseButton button) {
		pt += offset;

		if (down) {
			clicked = nodeAt(pt);
			if (clicked) {
				if (from) {
					saveUndo();
					if (edgeFrom == DragFrom:left) {
						from.left = clicked;
					} else {
						from.right = clicked;
					}
				}
				mouseOrigin = pt - clicked.at;
			} else {
				mouseOrigin = pt;
			}
			from = null;
		}
	}

	Bool mouseMoved(Point pt, Bool mouseDown) {
		pt += offset;

		if (from) {
			mouseOrigin = pt;
			return true;
		}

		if (mouseDown) {
			if (clicked) {
				clicked.at = pt - mouseOrigin;
				return true;
			} else {
				offset -= pt - mouseOrigin;
				return true;
			}
		}
		false;
	}

	void insert(Nat number, Point at) {
		at += offset;
		saveUndo();
		nodes << Node(at, number.toS);
	}

	void insertPipe(Point at) {
		at += offset;

		saveUndo();
		nodes << Node(at, "|");
	}

	void delete(Point at) {
		at += offset;

		saveUndo();

		for (i, n in nodes) {
			if (n.contains(at)) {
				for (k in nodes) {
					if (k.left is n)
						k.left = null;
					if (k.right is n)
						k.right = null;
				}

				nodes.remove(i);
				break;
			}
		}
	}

	void clear() {
		nodes = Node[];
	}

	void clear(Point at) {
		at += offset;

		if (n = nodeAt(at)) {
			n.left = null;
			n.right = null;
		}
	}

	void startDrag(Point pt, DragFrom edge) {
		pt += offset;

		if (n = nodeAt(pt)) {
			from = n;
			edgeFrom = edge;
		}
	}

	void abort() {
		from = null;
	}

	void load(SimpleNode node) {
		clear();
		offset = Point();
		undoStack.clear();

		Node n = loadI(node);
		n.at = Point(20, 20);
		layoutChildren(n);
	}

	private Node loadI(SimpleNode node) {
		Node n(Point(), node.text);
		nodes << n;

		if (left = node.left)
			n.left = loadI(left);
		if (right = node.right)
			n.right = loadI(right);

		n;
	}

	void rotateLeft(Point pt) {
		pt += offset;

		if (n = nodeAt(pt)) {
			if (child = n.left) {
				saveUndo();

				n.left = child.right;

				for (node in nodes) {
					node.replace(n, child);
				}

				child.right = n;
			}
		}
	}

	void rotateRight(Point pt) {
		pt += offset;

		if (n = nodeAt(pt)) {
			if (child = n.right) {
				saveUndo();

				n.right = child.left;

				for (node in nodes) {
					node.replace(n, child);
				}

				child.left = n;
			}
		}
	}

	void layoutNode(Point pt) {
		pt += offset;

		if (n = nodeAt(pt)) {
			var delta = layoutChildren(n);
			moveTree(n, -delta.left);
		}
	}

	private value LayoutDim {
		Float left;
		Float width;

		init(Float left, Float width) {
			init { left = left; width = width; }
		}
	}

	private LayoutDim layoutChildren(Node node) {
		Float prevLeft = 0;

		Size textSz = node.text.size;
		Float space = textSz.h * 0.2;

		Float height = textSz.h * 2;
		Float width = textSz.w;


		if (left = node.left) {
			left.at.x = node.at.x;
			left.at.y = node.at.y + height;
			var dim = layoutChildren(left);

			prevLeft = dim.width;
			node.at.x += dim.width + space;
			width += dim.width + space;
		}

		if (right = node.right) {
			right.at.x = node.at.x + textSz.w + space;
			right.at.y = node.at.y + height;
			var dim = layoutChildren(right);
			width += dim.width + space;
		}

		LayoutDim(prevLeft, width);
	}

	private void moveTree(Node node, Float delta) {
		node.at.x += delta;
		if (left = node.left)
			moveTree(left, delta);
		if (right = node.right)
			moveTree(right, delta);
	}

	void cloneTree(Point pt) {
		pt += offset;
		unless (n = nodeAt(pt))
			return;

		saveUndo();

		Float maxY = 0;
		for (n in nodes) {
			maxY = max(maxY, n.bottomRight.y);
		}

		Float delta = maxY + 30 - n.at.y;

		cloneTree(n, delta);
	}

	private Node cloneTree(Node n, Float dy) {
		Node c(n.at + Point(0, dy), n.text.text);
		nodes << c;

		if (left = n.left)
			c.left = cloneTree(left, dy);
		if (right = n.right)
			c.right = cloneTree(right, dy);

		c;
	}

	void deleteTree(Point pt) {
		pt += offset;
		unless (n = nodeAt(pt))
			return;

		saveUndo();

		Set<Node> toDelete;
		deleteTree(n, toDelete);

		for (Nat n = nodes.count; n > 0; n--) {
			Node check = nodes[n - 1];
			if (toDelete.has(check))
				nodes.remove(n - 1);

			if (l = check.left)
				if (toDelete.has(l))
					check.left = null;

			if (r = check.right)
				if (toDelete.has(r))
					check.right = null;
		}
	}

	private void deleteTree(Node n, Set<Node> toDelete) {
		toDelete.put(n);

		if (left = n.left)
			deleteTree(left, toDelete);
		if (right = n.right)
			deleteTree(right, toDelete);
	}

	Node? nodeAt(Point at) {
		for (n in nodes) {
			if (n.contains(at))
				return n;
		}
		null;
	}

	void undo() {
		if (undoStack.empty)
			return;

		nodes = undoStack.last;
		undoStack.pop();
	}

	void saveUndo() {
		// Keep from growing too much.
		if (undoStack.count > 20)
			undoStack.remove(0);

		Node->Node idMap;
		Node[] copy;
		for (i, n in nodes) {
			Node c = Node(n);
			idMap.put(n, c);
			copy << c;
		}

		for (c in copy) {
			c.left = update(c.left, idMap);
			c.right = update(c.right, idMap);
		}

		undoStack << copy;
	}

	private Node? update(Node? n, Node->Node idMap) : static {
		unless (n)
			return null;

		idMap.get(n, n);
	}
}


void main() {
	TreeWindow win;
	win.waitForClose();
}
