use nettle_sys::{
    nettle_hash, nettle_sha3_256, nettle_sha3_256_digest, nettle_sha3_256_init,
    nettle_sha3_256_update, sha3_256_ctx,
};
use std::default::Default;
use std::mem::zeroed;

use crate::hash::NettleHash;
use crate::hash::Hash;

/// 256 bit variant of the Secure Hash Algorithm 3 (SHA-3) defined in FIPS 202.
#[allow(non_camel_case_types)]
pub struct Sha3_256 {
    context: sha3_256_ctx,
}

impl_write_for_hash!(Sha3_256);

impl Clone for Sha3_256 {
    fn clone(&self) -> Self {
        use std::intrinsics::copy_nonoverlapping;

        unsafe {
            let mut ctx: sha3_256_ctx = zeroed();
            copy_nonoverlapping(&self.context, &mut ctx, 1);

            Sha3_256 { context: ctx }
        }
    }
}

impl Default for Sha3_256 {
    fn default() -> Self {
        let mut ctx = unsafe { zeroed() };

        unsafe {
            nettle_sha3_256_init(&mut ctx as *mut _);
        }

        Sha3_256 { context: ctx }
    }
}

impl Hash for Sha3_256 {
    fn digest_size(&self) -> usize {
        ::nettle_sys::SHA3_256_DIGEST_SIZE as usize
    }

    fn update(&mut self, data: &[u8]) {
        unsafe {
            nettle_sha3_256_update(
                &mut self.context as *mut _,
                data.len(),
                data.as_ptr(),
            );
        }
    }

    fn digest(&mut self, digest: &mut [u8]) {
        unsafe {
            nettle_sha3_256_digest(
                &mut self.context as *mut _,
                digest.len(),
                digest.as_mut_ptr(),
            );
        }
    }

    fn box_clone(&self) -> Box<Hash> {
        Box::new(self.clone())
    }
}

impl NettleHash for Sha3_256 {
    type Context = sha3_256_ctx;

    unsafe fn nettle_hash() -> &'static nettle_hash {
        &nettle_sha3_256
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn do_nothing() {
        let _ = Sha3_256::default();
    }

    //  CAVS 19.0
    //  "SHA3-256 ShortMsg" information for "SHA3AllBytes1-28-16"
    //  Length values represented in bits
    //  Generated on Thu Jan 28 13:32:44 2016
    #[test]
    fn nist_cavs_short_msg() {
        let mut ctx = Sha3_256::default();
        let mut digest = vec![0u8; ctx.digest_size()];

        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xa7\xff\xc6\xf8\xbf\x1e\xd7\x66\x51\xc1\x47\x56\xa0\x61\xd6\x62\xf5\x80\xff\x4d\xe4\x3b\x49\xfa\x82\xd8\x0a\x4b\x80\xf8\x43\x4a"[..]);

        ctx.update(b"\xe9");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xf0\xd0\x4d\xd1\xe6\xcf\xc2\x9a\x44\x60\xd5\x21\x79\x68\x52\xf2\x5d\x9e\xf8\xd2\x8b\x44\xee\x91\xff\x5b\x75\x9d\x72\xc1\xe6\xd6"[..]);

        ctx.update(b"\xd4\x77");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x94\x27\x9e\x8f\x5c\xcd\xf6\xe1\x7f\x29\x2b\x59\x69\x8a\xb4\xe6\x14\xdf\xe6\x96\xa4\x6c\x46\xda\x78\x30\x5f\xc6\xa3\x14\x6a\xb7"[..]);

        ctx.update(b"\xb0\x53\xfa");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x9d\x0f\xf0\x86\xcd\x0e\xc0\x6a\x68\x2c\x51\xc0\x94\xdc\x73\xab\xdc\x49\x20\x04\x29\x23\x44\xbd\x41\xb8\x2a\x60\x49\x8c\xcf\xdb"[..]);

        ctx.update(b"\xe7\x37\x21\x05");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x3a\x42\xb6\x8a\xb0\x79\xf2\x8c\x4c\xa3\xc7\x52\x29\x6f\x27\x90\x06\xc4\xfe\x78\xb1\xeb\x79\xd9\x89\x77\x7f\x05\x1e\x40\x46\xae"[..]);

        ctx.update(b"\x02\x96\xf2\xc4\x0a");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x53\xa0\x18\x93\x72\x21\x08\x1d\x09\xed\x04\x97\x37\x7e\x32\xa1\xfa\x72\x40\x25\xdf\xdc\x18\x71\xfa\x50\x3d\x54\x5d\xf4\xb4\x0d"[..]);

        ctx.update(b"\xe6\xfd\x42\x03\x7f\x80");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x22\x94\xf8\xd3\x83\x4f\x24\xaa\x90\x37\xc4\x31\xf8\xc2\x33\xa6\x6a\x57\xb2\x3f\xa3\xde\x10\x53\x0b\xbb\x69\x11\xf6\xe1\x85\x0f"[..]);

        ctx.update(b"\x37\xb4\x42\x38\x5e\x05\x38");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xcf\xa5\x50\x31\xe7\x16\xbb\xd7\xa8\x3f\x21\x57\x51\x30\x99\xe2\x29\xa8\x88\x91\xbb\x89\x9d\x9c\xcd\x31\x71\x91\x81\x99\x98\xf8"[..]);

        ctx.update(b"\x8b\xca\x93\x1c\x8a\x13\x2d\x2f");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xdb\xb8\xbe\x5d\xec\x1d\x71\x5b\xd1\x17\xb2\x45\x66\xdc\x3f\x24\xf2\xcc\x0c\x79\x97\x95\xd0\x63\x8d\x95\x37\x48\x1e\xf1\xe0\x3e"[..]);

        ctx.update(b"\xfb\x8d\xfa\x3a\x13\x2f\x98\x13\xac");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xfd\x09\xb3\x50\x18\x88\x44\x5f\xfc\x8c\x3b\xb9\x5d\x10\x64\x40\xce\xee\x46\x94\x15\xfc\xe1\x47\x47\x43\x27\x30\x94\x30\x6e\x2e"[..]);

        ctx.update(b"\x71\xfb\xac\xdb\xf8\x54\x17\x79\xc2\x4a");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xcc\x4e\x5a\x21\x6b\x01\xf9\x87\xf2\x4a\xb9\xca\xd5\xeb\x19\x6e\x89\xd3\x2e\xd4\xaa\xc8\x5a\xcb\x72\x7e\x18\xe4\x0c\xee\xf0\x0e"[..]);

        ctx.update(b"\x7e\x8f\x1f\xd1\x88\x2e\x4a\x7c\x49\xe6\x74");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x79\xbe\xf7\x8c\x78\xaa\x71\xe1\x1a\x33\x75\x39\x4c\x25\x62\x03\x7c\xd0\xf8\x2a\x03\x3b\x48\xa6\xcc\x93\x2c\xc4\x33\x58\xfd\x9e"[..]);

        ctx.update(b"\x5c\x56\xa6\xb1\x8c\x39\xe6\x6e\x1b\x7a\x99\x3a");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xb6\x97\x55\x6c\xb3\x0d\x6d\xf4\x48\xee\x38\xb9\x73\xcb\x69\x42\x55\x9d\xe4\xc2\x56\x7b\x15\x56\x24\x01\x88\xc5\x5e\xc0\x84\x1c"[..]);

        ctx.update(b"\x9c\x76\xca\x5b\x6f\x8d\x12\x12\xd8\xe6\x89\x6a\xd8");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x69\xdf\xc3\xa2\x58\x65\xf3\x53\x5f\x18\xb4\xa7\xbd\x9c\x0c\x69\xd7\x84\x55\xf1\xfc\x1f\x4b\xf4\xe2\x9f\xc8\x2b\xf3\x28\x18\xec"[..]);

        ctx.update(b"\x68\x7f\xf7\x48\x5b\x7e\xb5\x1f\xe2\x08\xf6\xff\x9a\x1b");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xfe\x7e\x68\xae\x3e\x1a\x91\x94\x4e\x4d\x1d\x21\x46\xd9\x36\x0e\x53\x33\xc0\x99\xa2\x56\xf3\x71\x1e\xdc\x37\x2b\xc6\xee\xb2\x26"[..]);

        ctx.update(
            b"\x41\x49\xf4\x1b\xe1\xd2\x65\xe6\x68\xc5\x36\xb8\x5d\xde\x41",
        );
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x22\x9a\x77\x02\x44\x8c\x64\x0f\x55\xda\xfe\xd0\x8a\x52\xaa\x0b\x11\x39\x65\x7b\xa9\xfc\x4c\x5e\xb8\x58\x7e\x17\x4e\xcd\x9b\x92"[..]);

        ctx.update(
            b"\xd8\x3c\x72\x1e\xe5\x1b\x06\x0c\x5a\x41\x43\x8a\x82\x21\xe0\x40",
        );
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xb8\x7d\x9e\x47\x22\xed\xd3\x91\x87\x29\xde\xd9\xa6\xd0\x3a\xf8\x25\x69\x98\xee\x08\x8a\x1a\xe6\x62\xef\x4b\xca\xff\x14\x2a\x96"[..]);

        ctx.update(b"\x26\x6e\x8c\xbd\x3e\x73\xd8\x0d\xf2\xa4\x9c\xfd\xaf\x0d\xc3\x9c\xd1");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x6c\x2d\xe3\xc9\x59\x00\xa1\xbc\xec\x6b\xd4\xca\x78\x00\x56\xaf\x4a\xcf\x3a\xa3\x6e\xe6\x40\x47\x4b\x6e\x87\x01\x87\xf5\x93\x61"[..]);

        ctx.update(b"\xa1\xd7\xce\x51\x04\xeb\x25\xd6\x13\x1b\xb8\xf6\x6e\x1f\xb1\x3f\x35\x23");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xee\x90\x62\xf3\x97\x20\xb8\x21\xb8\x8b\xe5\xe6\x46\x21\xd7\xe0\xca\x02\x6a\x9f\xe7\x24\x8d\x78\x15\x0b\x14\xbd\xba\xa4\x0b\xed"[..]);

        ctx.update(b"\xd7\x51\xcc\xd2\xcd\x65\xf2\x7d\xb5\x39\x17\x69\x20\xa7\x00\x57\xa0\x8a\x6b");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x7a\xac\xa8\x0d\xbe\xb8\xdc\x36\x77\xd1\x8b\x84\x79\x59\x85\x46\x36\x50\xd7\x2f\x25\x43\xe0\xec\x70\x9c\x9e\x70\xb8\xcd\x7b\x79"[..]);

        ctx.update(b"\xb3\x2d\xec\x58\x86\x5a\xb7\x46\x14\xea\x98\x2e\xfb\x93\xc0\x8d\x9a\xcb\x1b\xb0");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x6a\x12\xe5\x35\xdb\xfd\xda\xb6\xd3\x74\x05\x8d\x92\x33\x8e\x76\x0b\x1a\x21\x14\x51\xa6\xc0\x9b\xe9\xb6\x1e\xe2\x2f\x3b\xb4\x67"[..]);

        ctx.update(b"\x4e\x0c\xc4\xf5\xc6\xdc\xf0\xe2\xef\xca\x1f\x9f\x12\x93\x72\xe2\xdc\xbc\xa5\x7e\xa6");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xd2\xb7\x71\x78\x64\xe9\x43\x8d\xd0\x2a\x4f\x8b\xb0\x20\x3b\x77\xe2\xd3\xcd\x8f\x8f\xfc\xf9\xdc\x68\x4e\x63\xde\x5e\xf3\x9f\x0d"[..]);

        ctx.update(b"\xd1\x6d\x97\x8d\xfb\xae\xcf\x2c\x8a\x04\x09\x0f\x6e\xeb\xdb\x42\x1a\x5a\x71\x11\x37\xa6");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x7f\x49\x79\x13\x31\x8d\xef\xdc\x60\xc9\x24\xb3\x70\x4b\x65\xad\xa7\xca\x3b\xa2\x03\xf2\x3f\xb9\x18\xc6\xfb\x03\xd4\xb0\xc0\xda"[..]);

        ctx.update(b"\x47\x24\x9c\x7c\xb8\x5d\x8f\x02\x42\xab\x24\x0e\xfd\x16\x4b\x9c\x8b\x0b\xd3\x10\x4b\xba\x3b");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x43\x5e\x27\x6f\x06\xae\x73\xaa\x5d\x5d\x60\x18\xf5\x8e\x0f\x00\x9b\xe3\x51\xea\xda\x47\xb6\x77\xc2\xf7\xc0\x64\x55\xf3\x84\xe7"[..]);

        ctx.update(b"\xcf\x54\x9a\x38\x3c\x0a\xc3\x1e\xae\x87\x0c\x40\x86\x7e\xeb\x94\xfa\x1b\x6f\x3c\xac\x44\x73\xf2");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xcd\xfd\x1a\xfa\x79\x3e\x48\xfd\x0e\xe5\xb3\x4d\xfc\x53\xfb\xce\xe4\x3e\x9d\x2a\xc2\x15\x15\xe4\x74\x64\x75\x45\x3a\xb3\x83\x1f"[..]);

        ctx.update(b"\x9b\x3f\xdf\x8d\x44\x86\x80\x84\x0d\x62\x84\xf2\x99\x7d\x3a\xf5\x5f\xfd\x85\xf6\xf4\xb3\x3d\x7f\x8d");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x25\x00\x5d\x10\xe8\x4f\xf9\x7c\x74\xa5\x89\x01\x3b\xe4\x2f\xb3\x7f\x68\xdb\x64\xbd\xfc\x76\x26\xef\xc0\xdd\x62\x80\x77\x49\x3a"[..]);

        ctx.update(b"\x6b\x22\xfe\x94\xbe\x2d\x0b\x25\x28\xd9\x84\x7e\x12\x7e\xb6\xc7\xd6\x96\x7e\x7e\xc8\xb9\x66\x0e\x77\xcc");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x15\x7a\x52\xb0\x47\x76\x39\xb3\xbc\x17\x96\x67\xb3\x5c\x1c\xdf\xbb\x3e\xef\x84\x5e\x44\x86\xf0\xf8\x4a\x52\x6e\x94\x0b\x51\x8c"[..]);

        ctx.update(b"\xd8\xde\xca\xfd\xad\x37\x79\x04\xa2\x78\x95\x51\x13\x5e\x78\x2e\x30\x2a\xed\x84\x50\xa4\x2c\xfb\x89\x60\x0c");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x3d\xde\xcf\x5b\xba\x51\x64\x3c\xd7\x7e\xbd\xe2\x14\x1c\x85\x45\xf8\x62\x06\x7b\x20\x99\x90\xd4\xcb\x65\xbf\xa6\x5f\x4f\xa0\xc0"[..]);

        ctx.update(b"\x93\x8f\xe6\xaf\xdb\xf1\x4d\x12\x29\xe0\x35\x76\xe5\x32\xf0\x78\x89\x87\x69\xe2\x06\x20\xae\x21\x64\xf5\xab\xfa");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x95\x11\xab\xd1\x3c\x75\x67\x72\xb8\x52\x11\x45\x78\xef\x9b\x96\xf9\xdc\x7d\x0f\x2b\x8d\xcd\xe6\xea\x7d\x1b\xd1\x4c\x51\x88\x90"[..]);

        ctx.update(b"\x66\xeb\x5e\x73\x96\xf5\xb4\x51\xa0\x2f\x39\x69\x9d\xa4\xdb\xc5\x05\x38\xfb\x10\x67\x8e\xc3\x9a\x5e\x28\xba\xa3\xc0");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x54\x0a\xcf\x81\x81\x0a\x19\x99\x96\xa6\x12\xe8\x85\x78\x13\x08\x80\x2f\xe4\x60\xe9\xc6\x38\xcc\x02\x2e\x17\x07\x6b\xe8\x59\x7a"[..]);

        ctx.update(b"\xde\x98\x96\x8c\x8b\xd9\x40\x8b\xd5\x62\xac\x6e\xfb\xca\x2b\x10\xf5\x76\x9a\xac\xaa\x01\x36\x57\x63\xe1\xb2\xce\x80\x48");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x6b\x2f\x25\x47\x78\x14\x49\xd4\xfa\x15\x81\x80\xa1\x78\xef\x68\xd7\x05\x61\x21\xbf\x8a\x2f\x2f\x49\x89\x1a\xfc\x24\x97\x85\x21"[..]);

        ctx.update(b"\x94\x46\x4e\x8f\xaf\xd8\x2f\x63\x0e\x6a\xab\x9a\xa3\x39\xd9\x81\xdb\x0a\x37\x2d\xc5\xc1\xef\xb1\x77\x30\x59\x95\xae\x2d\xc0");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xea\x79\x52\xad\x75\x96\x53\xcd\x47\xa1\x80\x04\xac\x2d\xbb\x9c\xf4\xa1\xe7\xbb\xa8\xa5\x30\xcf\x07\x05\x70\xc7\x11\xa6\x34\xea"[..]);

        ctx.update(b"\xc1\x78\xce\x0f\x72\x0a\x6d\x73\xc6\xcf\x1c\xaa\x90\x5e\xe7\x24\xd5\xba\x94\x1c\x2e\x26\x28\x13\x6e\x3a\xad\x7d\x85\x37\x33\xba");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x64\x53\x7b\x87\x89\x28\x35\xff\x09\x63\xef\x9a\xd5\x14\x5a\xb4\xcf\xce\x5d\x30\x3a\x0c\xb0\x41\x5b\x3b\x03\xf9\xd1\x6e\x7d\x6b"[..]);

        ctx.update(b"\x6e\xf7\x0a\x3a\x21\xf9\xf7\xdc\x41\xc5\x53\xc9\xb7\xef\x70\xdb\x82\xca\x69\x94\xac\x89\xb3\x62\x7d\xa4\xf5\x21\xf0\x7e\x1a\xe2\x63");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x0a\xfe\x03\xb1\x75\xa1\xc9\x48\x96\x63\xd8\xa6\xf6\x6d\x1b\x24\xab\xa5\x13\x9b\x99\x64\x00\xb8\xbd\x3d\x0e\x1a\x79\x58\x0e\x4d"[..]);

        ctx.update(b"\x0c\x4a\x93\x1f\xf7\xea\xce\x5e\xa7\xcd\x8d\x2a\x67\x61\x94\x08\x38\xf3\x0e\x43\xc5\xd1\x25\x32\x99\xab\xd1\xbd\x90\x3f\xed\x1e\x8b\x36");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xdc\x5b\xeb\xe0\x5c\x49\x94\x96\xa7\xeb\xfe\x04\x30\x9c\xae\x51\x5e\x3e\xa5\x7c\x5d\x2a\x5f\xe2\xe6\x80\x12\x43\xdd\x52\xc9\x3b"[..]);

        ctx.update(b"\x21\x0f\x7b\x00\xbf\x8b\x43\x37\xb4\x24\x50\xc7\x21\xc3\xf7\x81\x25\x63\x59\xd2\x08\x73\x38\x46\xb9\x7c\x0a\x4b\x7b\x04\x4c\x38\xdb\xb2\x19");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x33\x05\xc9\xd2\x8e\x05\x28\x8a\x2d\x13\x99\x4d\x64\xc8\x8d\x35\x06\x39\x9c\xd6\x2b\x2b\x54\x42\x13\xcf\x35\x39\xa8\xe9\x2e\x2e"[..]);

        ctx.update(b"\x3c\xb8\x99\x27\x59\xe2\xdc\x60\xeb\xb0\x22\xbd\x8e\xe2\x7f\x0f\x98\x03\x9e\x6a\x9f\xe3\x60\x37\x3b\x48\xc7\x85\x0c\xe1\x13\xa0\xff\x7b\x2a\xe5");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x3c\x00\xbf\x3e\x12\xad\xe9\xd2\xde\x27\x56\x50\x6f\x80\x9f\x14\x7c\x8d\x6a\xdc\x22\xe7\xbb\x66\x6e\x0b\x1d\x26\x46\x9e\x65\xa5"[..]);

        ctx.update(b"\x22\x63\x4f\x6b\xa7\xb4\xfc\xca\xa3\xba\x40\x40\xb6\x64\xdb\xe5\xa7\x2b\xf3\x94\xfb\x53\x4e\x49\xc7\x6e\xc4\xcd\xc2\x23\xf4\x96\x9e\x2d\x37\xe8\x99");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xa8\x7e\x5c\x78\x83\x7d\x7b\xe0\x06\x0d\x8f\x5e\xda\x97\x54\x89\xec\x96\x1b\x28\xd7\x08\x8f\x42\xa7\x0f\x92\x41\x4a\xe1\x77\x93"[..]);

        ctx.update(b"\x6e\x1d\xcd\x79\x6b\x20\x15\xee\x67\x60\xf9\x8f\xdb\x40\xe6\x68\xb2\xcf\x38\xb0\x5c\x91\xf6\xa9\x1e\x83\xbc\xc8\xac\x59\xf8\x16\xf9\x0a\x59\xd6\x4e\x8e");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x74\x6b\xf8\x45\xc0\x8a\xa1\x86\xb5\xfe\x1c\xa3\x55\x28\x23\x2c\x4a\x49\x1a\x3a\x2a\x32\xcd\x23\xe9\x90\xbc\x60\x3f\x32\x68\xae"[..]);

        ctx.update(b"\xee\x0b\xe2\x03\x20\xf9\xd4\x40\x73\x28\x12\x65\xa6\xe9\xfa\x6b\x9d\x25\x24\x95\x62\x4b\x8d\x01\x6b\x8e\xf5\x7e\x1b\x4e\x85\x9d\x8a\xd3\xb5\x0b\x89\x41\x6d");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xa3\x25\x7b\xaf\x14\xca\x16\xe1\x13\x7d\xc5\x15\x87\x03\xf3\xb0\x2e\xbc\x74\xfc\x76\x77\x16\x5f\xe8\x6d\x4b\xe1\xf3\x8e\x2f\x7c"[..]);

        ctx.update(b"\x8a\xe2\xda\x24\x26\x35\xb6\x56\x82\x89\xbf\x6b\xec\x8a\x43\x8d\xba\xc1\xf5\xb4\xd5\x0a\x90\xbb\x74\x49\xbd\xb9\x2a\x59\x37\x8e\x23\x45\x2d\xbc\xab\xbb\xe8\x79");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xe2\x5c\x44\x80\x2c\x5c\xf2\xe9\xf6\x33\xe6\x83\xd3\x7a\xa8\xc8\xdb\x8a\x0e\x21\xc3\x67\x80\x81\x21\xd1\x4d\x96\xc8\xa4\x00\xb5"[..]);

        ctx.update(b"\xbd\xd0\x25\x2d\xec\x5b\x79\x8e\xf2\x0e\x51\x79\x1a\x18\xe8\xca\x23\x4d\x9b\xfd\xe6\x32\xa9\xe5\x39\x53\x37\xa1\x12\xdd\x97\xcd\xf0\x68\xc9\xf5\x76\x15\x42\x4f\x59");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xe0\x2c\x1b\x19\x79\x79\xc4\x4a\x5a\x50\xd0\x5e\xa4\x88\x2c\x16\xd8\x20\x5c\x2e\x33\x44\x26\x5f\x8f\xe0\xe8\x0a\xed\x06\xc0\x65"[..]);

        ctx.update(b"\xc4\xc7\xb6\x31\x5c\xb6\x0b\x0e\x6c\xd0\x1e\xf0\xb6\x5f\x64\x86\xfd\xae\x4b\x94\xc6\xbe\x21\x46\x5c\x3a\x31\xc4\x16\xad\x2f\x06\xdc\xf3\xd6\xea\xe8\xee\xcf\x84\xca\x7a");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x2d\xa2\x18\x67\xcd\x6b\x54\x02\xd3\xca\xff\x92\xa0\x5f\xdd\xfc\xa9\x01\x99\xfd\x51\xa9\x4a\x06\x6a\xf1\x64\xce\x3d\x36\xc9\x49"[..]);

        ctx.update(b"\xb1\x79\x77\xac\xed\x3a\x11\x84\xb1\x4b\x0e\x41\xa0\x4d\xd8\xb5\x13\xc9\x25\xca\x19\x21\x1e\x1a\xbd\xc6\xc1\xb9\x87\xac\x84\x55\x45\xfb\x3b\x82\x0a\x08\x3b\x4f\x78\x83\xc0");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xf9\x1b\x01\x6d\x01\x3e\xde\x8d\x6a\x2e\x1e\xfd\x4c\x0d\xd9\x94\x17\xda\x8b\x02\x22\xd7\x87\x86\x7c\xa0\x2b\x0e\xa2\xe8\x0e\x45"[..]);

        ctx.update(b"\xf6\x5c\x3a\xa1\xd9\x98\x1a\x84\xe4\x9f\xc8\x6d\x93\x8f\x3f\x75\x6f\x60\xe3\x85\x8d\x5e\x1f\x69\x57\xdd\x4d\x26\x8e\x28\xd6\x8e\x90\xba\x9a\x11\xd7\xb1\x92\xd6\xc3\x7f\xb3\x0b");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x3a\xcb\xeb\xf8\xed\xa9\xd3\xc9\x9a\x6b\x6b\x66\x63\x66\xc3\x91\xe8\x20\x0d\x55\xfd\x33\xad\x86\x80\x73\x4d\xef\x1d\xc7\xae\x85"[..]);

        ctx.update(b"\x49\xab\xba\x1f\xa9\x8f\x3c\x44\x70\xd5\xdd\x4e\xd3\x69\x24\xaf\x4a\x7a\xd6\x2f\x4c\x2d\xd1\x3e\x59\x92\x38\x88\x3e\xd7\xd0\xcb\x95\xbb\xaa\xe5\x8b\x46\x03\x32\xe6\xb7\x68\x14\x46");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x02\xbc\xd9\xea\x4f\x1a\xa5\x27\x6f\x38\xe3\x03\x51\xa1\x4a\x07\x2b\xc5\xd5\x3a\x52\xd0\x4d\x55\x9a\x65\xca\x46\xf1\xbc\xb5\x6e"[..]);

        ctx.update(b"\x27\x56\x45\xb5\xa2\x51\x4f\xe6\x5a\x82\xef\xac\x57\xe4\x06\xf2\x24\xe0\x25\x96\x77\x67\x4f\x1d\x13\x3f\x00\xa5\xee\x9a\x6d\x1a\x8f\xed\x0e\xad\xbb\xff\x5a\x82\x50\x41\xd2\xa9\x71\x5d");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xc7\x0a\x87\x4d\x78\x6c\xd0\xf3\xf0\x9f\xa4\xdc\x1b\xb8\xf5\x51\xd4\x5f\x26\xd7\x7a\xd6\x3d\xe1\xa9\xfd\xfb\x3b\x7c\x09\xc0\x41"[..]);

        ctx.update(b"\xcd\x02\xb3\x21\x07\xb9\xa6\x40\xfc\x1b\xf4\x39\xac\x81\xa5\xc2\x7d\x03\x7c\x60\x76\xe1\xcf\xe6\xad\x22\x96\x38\x03\x7a\xc1\x55\x0e\x71\xcf\x95\x57\xc2\x9c\x2f\xc6\x01\x7a\xfd\x5a\x81\x84");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x36\xc7\x3d\x11\xd4\x50\x78\x4e\xb9\x9a\xf0\x68\xcd\x4e\x1c\xbc\x57\x68\xc8\xa2\x11\x80\x10\xac\xee\xc6\xd8\x52\xdd\xa8\x0d\x95"[..]);

        ctx.update(b"\x5a\x72\xe0\xe1\xae\xc8\x2a\x65\x41\xf0\x48\x83\xbb\x46\x3b\x0c\x39\xc2\x2b\x59\x43\x1c\xfb\x8b\xfd\x33\x21\x17\xa1\xaf\xb5\x83\x2c\xe5\xc7\x6a\x58\xfc\xf6\xc6\xcb\x4e\x3e\x6f\x8e\x11\x12\xde");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x90\xfc\x31\x93\x55\x2e\xc7\x1d\x33\x15\xeb\xbb\x80\x79\x13\xaf\xd4\xcd\x2f\x08\x33\xa6\x5e\x40\xd0\x11\xd6\x4d\xe5\xe6\x65\x13"[..]);

        ctx.update(b"\x43\x40\x21\x65\x91\x18\x90\x71\x9f\x91\x79\xf8\x83\xbb\xbc\x2a\x3b\xe7\x76\x82\xe6\x0d\xd2\x4b\x35\x6a\x22\x62\x1c\x6d\x2e\x3d\xcd\xd4\xcb\x2c\xe6\x13\xb0\xdf\xe9\xf5\x86\x29\xee\x85\x3e\x03\x94");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x5c\x4b\x6c\xea\xc9\x44\x1d\xef\xa9\x9b\x10\xb8\x05\xa7\x25\xd4\x01\x8b\x74\xb3\xe1\xf2\x4a\xd8\x93\x4f\xc8\x9b\x41\xb8\xfd\x9e"[..]);

        ctx.update(b"\xfc\x56\xca\x9a\x93\x98\x2a\x46\x69\xcc\xab\xa6\xe3\xd1\x84\xa1\x9d\xe4\xce\x80\x0b\xb6\x43\xa3\x60\xc1\x45\x72\xae\xdb\x22\x97\x4f\x0c\x96\x6b\x85\x9d\x91\xad\x5d\x71\x3b\x7a\xd9\x99\x35\x79\x4d\x22");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xe2\x18\x06\xce\x76\x6b\xbc\xe8\xb8\xd1\xb9\x9b\xcf\x16\x2f\xd1\x54\xf5\x46\x92\x35\x1a\xec\x8e\x69\x14\xe1\xa6\x94\xbd\xa9\xee"[..]);

        ctx.update(b"\xac\xe6\x29\x7e\x50\xd5\x0a\x11\x38\x81\x18\xef\xc8\x8e\xf9\x72\x09\xb1\x1e\x9d\xfc\xb7\xad\x48\x2f\xc9\xbf\x7d\x8d\xee\xcc\x23\x7a\xd1\x63\xd9\x20\xc5\x1f\x25\x03\x06\xd6\xce\xdc\x41\x13\x86\xa4\x57\xc7");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xf5\x58\x14\x03\xa0\x82\xbb\xf5\xad\x7e\x09\xbd\xfc\xcc\x43\xbf\x96\x83\xeb\xc8\x82\x91\xd7\x1d\x9c\xe8\x85\xa3\x7e\x95\x2b\xd6"[..]);

        ctx.update(b"\x3b\xad\x18\x04\x6e\x94\x24\xde\x24\xe1\x29\x44\xcd\x99\x2c\xfb\xa4\x55\x6f\x0b\x2a\xe8\x8b\x7b\xd3\x42\xbe\x5c\xff\x95\x86\x09\x2b\xb6\x6f\xac\x69\xc5\x29\x04\x0d\x10\xdd\x66\xaa\x35\xc1\x02\x3d\x87\xeb\x68");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xfa\xed\x76\xff\x5a\x1c\xd9\x91\x83\xb3\x11\xe5\x02\xc5\x4e\x51\x6d\x70\xa8\x70\x50\xcf\x89\x61\xc8\xcd\x46\xf6\x5c\x13\x58\xcd"[..]);

        ctx.update(b"\xe5\x64\xc9\xa1\xf1\xaa\xf8\x54\x5a\x25\x9f\x52\xc3\xfd\x18\x21\xed\x03\xc2\x2f\xd7\x42\x4a\x0b\x2a\xd6\x29\xd5\xd3\x02\x6e\xf4\xf2\x7c\xbe\x06\xf3\x0b\x99\x1d\xfa\x54\xde\x28\x85\xf1\x92\xaf\x4d\xc4\xdd\xc4\x6d");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x81\x15\x29\xc6\x00\xc9\xd7\x80\xf7\x96\xa2\x9a\x6b\x3e\x89\xf8\xa1\x2b\x3f\x29\xc3\x6f\x72\xb0\x6c\xca\x7e\xdc\x36\xf4\x8d\xc0"[..]);

        ctx.update(b"\x60\x43\xfa\x64\x65\xd6\x9c\xab\x45\x52\x0a\xf5\xf0\xfd\x46\xc8\x1d\xbf\x67\x75\x31\x79\x98\x02\x62\x98\x63\x68\x1c\xea\x30\xff\xa3\xb0\x08\x36\xfb\xf4\x9f\x87\x05\x1d\x92\xaa\xea\xc0\xed\x09\xbc\xb9\xf0\x75\x5b\x7b");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xb0\xfc\xee\xcd\xae\xf6\xc7\x6d\x5f\xc3\x83\x5b\x52\x3c\xe2\x41\x6f\x4a\x9b\x9b\xd1\xf9\x02\x34\x44\x5d\xf0\xf2\xb6\x89\xf2\xf5"[..]);

        ctx.update(b"\x20\x40\xc5\x38\xc7\x92\x37\xe6\xf2\xb8\x18\x8c\x63\x75\xec\x2f\x61\x0a\xc2\x30\x16\x07\xb9\xc2\x36\x60\xc3\xa1\xe1\xc3\xa9\x02\xcb\x29\x50\xc5\x9a\xac\x3a\xf2\x8f\x98\x4f\x63\x69\xc4\xde\xbe\x86\x23\xdf\xa7\x4c\x96\x7b");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xe3\x3d\xbd\xc0\xac\xc2\x3f\xcf\xad\x3c\x75\x9c\x43\x33\x41\x0b\xd3\xa4\x0e\xfb\x13\x66\xad\xe1\x57\xd2\xc8\x1d\x65\xa0\xa6\xc7"[..]);

        ctx.update(b"\x00\xff\x6c\x96\xb7\xaa\x3c\xf2\x7d\x03\x6c\xf2\x0a\xf7\x03\x14\x34\x11\x32\x52\x57\x4b\xda\x9c\xf9\x24\x4d\x85\xae\xf2\x59\x3d\x3a\x7a\x83\xbf\xf6\xbe\x90\x4b\x75\x16\x4a\x17\x66\x82\x80\x42\xbc\x3f\x4f\x09\x0d\x98\xa0\x3d");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xd0\x00\xea\xfc\xa3\x48\x15\x78\x3b\xed\x9b\x05\x0c\x69\x01\xc9\x7f\x2e\x77\xd4\x77\x1a\x0e\xd7\x24\xdd\x8f\x6f\xf1\x44\x87\x91"[..]);

        ctx.update(b"\xe8\xdf\x14\x93\x6c\xce\x11\x81\x39\xe6\x90\xf1\x66\x2f\x88\xcf\xbc\x9c\x33\x3b\x6d\xea\x65\x8c\x02\xcb\x1d\x95\x96\x44\x59\x28\x42\x54\x2f\xd9\xd8\xd6\x1a\x04\xd4\xa8\x92\x12\x8f\x0d\xdf\xf7\xb6\x50\x2e\xff\xfb\xab\xe5\xcb\x0a");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x34\x79\xa9\x61\x7a\x3a\xdc\xa3\x58\x54\xc0\x8f\xe9\x87\xc2\xfe\x7f\xf2\xb0\x1b\x04\xf2\xd9\x52\xc1\x07\xb3\xf0\x66\x42\x05\x51"[..]);

        ctx.update(b"\x4e\xd9\x81\xa3\x1f\x70\xdd\x6b\x70\xc1\x61\xbe\x1f\x01\xfc\x1b\xba\x54\xd0\x6d\x94\x94\xe7\xeb\x19\x4e\x21\x3d\x5e\x0e\x71\xe0\xfd\xdd\x49\xcb\x1f\x07\x53\x53\xda\x22\x62\x4c\xbe\x4b\xa8\x71\xaa\xb3\x29\x06\xe4\x5b\x6f\xbb\x69\x1b");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x9c\x82\x4a\x00\xe0\x68\xd2\xfd\xa7\x3f\x9c\x2e\x77\x98\xe8\xd9\x39\x4f\x57\xf9\x4d\xf0\xed\xeb\x13\x2e\x78\xe8\xa3\x79\xa0\xcf"[..]);

        ctx.update(b"\x78\x02\xb7\x0c\x61\x58\xbc\x26\xd5\xf1\x57\x67\x1c\x3f\x3d\x81\xab\x39\x9d\xb5\x52\xb9\xf8\x51\xb7\x23\x33\x77\x03\x48\xeb\x1f\xdb\x8a\x08\x5f\x92\x40\x95\xeb\x9d\x5c\xcf\xd8\x47\x4b\x7b\xa5\xa6\x1c\x7d\x7b\xcd\xe5\xa7\xb4\x43\x62\xcf");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xfa\x97\x26\xcc\xb0\x68\xc0\xad\xb5\xd2\x00\x79\xc3\x5a\x31\x8b\x3d\x95\x1e\xb4\x3b\x19\x6c\x50\x9a\xb7\x90\xb7\xe9\x20\x22\x07"[..]);

        ctx.update(b"\xff\x83\xdc\xd7\xc1\xa4\x88\xe5\xa1\x28\xd5\xb7\x46\x28\x45\x52\xf1\xf2\xc0\x91\x61\x5d\x95\x19\xf4\x59\xbc\x90\x10\xca\x5e\x0a\xc1\x97\x96\xc4\xa3\xfd\x7a\x15\x03\x2a\x55\xa1\x41\x07\x37\xd0\x78\x55\xb0\x7f\x61\xfb\xd8\xf5\x75\x9e\x92\x18");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x8b\xd8\xd4\x94\xa4\x1a\xcd\xa4\xb7\xcd\x29\x94\xba\xda\xec\xff\x0f\x46\xba\x27\x43\x45\x8f\x6c\x3f\xdc\x02\x26\xf9\x49\x2e\xde"[..]);

        ctx.update(b"\xaf\xd4\x76\x4c\xc7\xd5\xde\x16\xa3\xcf\x80\xc5\x1d\x0c\x0d\x91\x9f\x18\x70\x0c\x7d\xc9\xbc\x4e\x88\x7d\x63\x4f\xe0\xa3\xaa\x94\x09\x7d\x59\x0e\x41\x23\xb7\x3f\x11\xcc\xb5\x9e\x23\x49\x6a\x3d\x53\xd2\xbf\xa9\x08\x05\x6c\x11\xc5\x2c\x23\xab\xfb");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xe9\xe3\xb3\xda\x64\x8c\xf2\x30\xf1\x97\x3f\x38\x14\xeb\x81\x31\x6d\x2a\x49\x68\x26\xea\x39\xad\xf4\x67\x45\x76\xf9\x7e\x11\x67"[..]);

        ctx.update(b"\x6f\xa6\xde\x50\x97\x19\xff\xbf\x17\x75\x9f\x05\x14\x53\xc0\xac\x3c\xbe\x13\x34\x65\x46\xbb\xc1\x70\x50\x54\x10\x74\xb0\x34\xaf\x19\x7a\xf0\x6e\x41\x14\x22\x11\xee\x90\x6a\x47\x60\x39\xb3\xe0\x7d\x6c\xb8\x3a\x76\xaa\xc6\xfc\xa8\xea\xc3\x07\xc0\x34");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x76\x66\x30\x99\x3f\xbb\x65\x1f\xd8\xd3\x60\x3e\x3e\xeb\xc8\x19\x31\xfb\x13\x02\xa4\x67\x91\xdf\x25\x9a\x6e\x13\xca\x2c\xba\x9f"[..]);

        ctx.update(b"\x93\xcb\xb7\xe4\x7c\x88\x59\xbe\xf9\x39\x15\x5b\xea\x48\x80\x90\x28\x3e\xcf\x50\x23\xd9\x97\x67\xc9\x60\xd8\x6b\xaa\x33\x3a\xf0\x5a\xa6\x96\xfc\x17\x0f\xb8\xbb\xac\x1e\x64\x73\x95\x6d\x96\xb9\x64\x58\x0e\xe6\x64\x0f\x0c\xc5\x7b\xe9\x59\x8e\x55\xfc\x86");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xd3\x21\x2a\xbc\xa1\x10\x0e\xb7\x65\x8c\x0f\x91\x6d\xaf\x26\x92\xc5\x7a\x47\xb7\x72\xee\x03\x1c\x4e\xc6\xad\x28\xa4\xa4\x6d\xe9"[..]);

        ctx.update(b"\x67\xe3\x84\xd2\x09\xf1\xbc\x44\x9f\xa6\x7d\xa6\xce\x5f\xbb\xe8\x4f\x46\x10\x12\x9f\x2f\x0b\x40\xf7\xc0\xca\xea\x7e\xd5\xcb\x69\xbe\x22\xff\xb7\x54\x1b\x20\x77\xec\x10\x45\x35\x6d\x9d\xb4\xee\x71\x41\xf7\xd3\xf8\x4d\x32\x4a\x5d\x00\xb3\x36\x89\xf0\xcb\x78");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x9c\x91\x60\x26\x86\x08\xef\x09\xfe\x0b\xd3\x92\x7d\x3d\xff\xa0\xc7\x34\x99\xc5\x28\x94\x3e\x83\x7b\xe4\x67\xb5\x0e\x5c\x1f\x1e"[..]);

        ctx.update(b"\x4b\xef\x1a\x43\xfa\xac\xc3\xe3\x84\x12\xc8\x75\x36\x06\x06\xa8\x11\x5d\x91\x97\xd5\x9f\x61\xa8\x5e\x0b\x48\xb4\x33\xdb\x27\x69\x5d\xc9\x62\xed\x75\xd1\x91\xc4\x01\x39\x79\xf4\x01\xcf\x3a\x67\xc4\x72\xc9\x90\x00\xd3\xa1\x52\x22\x7d\xb6\x1d\xe3\x13\xab\x5a\x1c");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x87\x03\xa1\xf7\x42\x4c\x35\x35\xf1\xd4\xf8\x8c\x9b\x03\xd1\x94\x89\x34\x99\x47\x89\x69\xfb\xb0\xa5\xdc\x28\x08\xa0\x69\xab\x8f"[..]);

        ctx.update(b"\xf0\xbe\x5e\x96\x1b\xb5\x5b\x3a\x94\x52\xa5\x36\x50\x4f\x61\x2a\x3e\x66\xae\xc8\x16\x0a\x88\x2e\x51\x56\xeb\x72\x78\x43\x3b\x7e\xa2\x1d\xe3\x1e\x39\x38\x3d\x57\xfc\xdf\xb2\xfb\x4a\x8d\x22\x7a\x9d\x60\x85\xfb\x55\xca\xd3\xab\xb7\x8a\x22\x55\x35\xda\x0e\x34\xef\xea");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x2f\xa1\x80\x20\x9b\xf6\xb4\xad\x13\xc3\x57\xd9\x17\xfa\xbb\x3e\x52\xc1\x01\xa0\xcd\xb3\xf2\x29\x9f\xa0\xf7\xf8\x1d\xfb\x84\x8e"[..]);

        ctx.update(b"\x20\x6f\x1c\x36\xba\x25\xae\xa7\x33\x98\xff\xfc\x9b\x65\xc4\x63\x7c\xc1\xf0\x5a\x6b\xbe\xe0\x14\xdc\xcb\xd6\x1e\x3b\x7a\xa9\x42\x38\x87\xbb\xac\x62\x15\x2a\x4b\xf7\x3a\x4b\x7a\xfa\xbe\x54\xe0\x87\x20\x58\x94\x64\xda\x79\x85\xd8\xe6\x59\x1a\xc0\x81\xd1\x15\xdf\x2f\xe6");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x55\x8e\xa7\xc8\x00\xb6\x87\x38\x0c\xce\x7e\x06\x00\x6e\x1e\xbe\x0b\x89\x97\x3f\x78\x8c\x4c\xaa\xc5\x78\x0f\x22\xdb\xf3\x82\xe8"[..]);

        ctx.update(b"\x8c\xd7\x14\x34\xc0\x06\x63\xf3\xbd\xa0\x20\x55\x08\xa4\xa2\x66\x54\x8d\xc6\x9e\x00\xca\x91\xfd\xe0\x6d\x16\x5b\x40\x27\x9a\xf9\x26\x74\xf7\x5b\xd8\x13\x3e\x5a\x9e\xb9\xa0\x75\xc9\x06\x8f\x68\xf4\xb8\x20\x00\x8a\x1f\xb4\x2d\x89\xd1\xd7\x59\x85\x9e\x68\xf8\xef\xc6\xfb\x60");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x08\x5b\x34\x3b\x08\x51\x6f\x32\x0a\x9b\x90\xfe\x50\x44\x0a\x8b\xc5\x1a\xe0\x85\x0f\xa3\x8d\x88\x72\x4a\x4d\x6b\xd3\xdf\x1a\xd4"[..]);

        ctx.update(b"\x4c\xf5\xbb\xd9\x1c\xac\x61\xc2\x11\x02\x05\x26\x34\xe9\x9f\xae\xdd\x6c\xdd\xdc\xd4\x42\x6b\x42\xb6\xa3\x72\xf2\x9a\x5a\x5f\x35\xf5\x1c\xe5\x80\xbb\x18\x45\xa3\xc7\xcf\xcd\x44\x7d\x26\x9e\x8c\xae\xb9\xb3\x20\xbb\x73\x1f\x53\xfe\x5c\x96\x9a\x65\xb1\x2f\x40\x60\x3a\x68\x5a\xfe");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xf9\xdb\xb8\x8c\x5b\xb4\x41\x5e\x17\xde\xe9\x22\x21\x74\x53\x8e\xea\xb3\x71\xb1\x2d\x8d\x57\x2c\xfd\xf5\x5b\x80\x6e\x31\x58\xe4"[..]);

        ctx.update(b"\xe0\x0e\x46\xc9\x6d\xec\x5c\xb3\x6c\xf4\x73\x20\x48\x37\x66\x57\xbc\xd1\xef\xf0\x8c\xcc\x05\xdf\x73\x41\x68\xae\x5c\xc0\x7a\x0a\xd5\xf2\x50\x81\xc0\x7d\x09\x8a\x4b\x28\x5e\xc6\x23\x40\x7b\x85\xe5\x3a\x0d\x8c\xd6\x99\x9d\x16\xd3\x13\x1c\x18\x8b\xef\xbf\xc9\xeb\xb1\x0d\x62\xda\xf9");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x35\x71\x32\x6a\x15\x77\xc4\x00\xb9\x67\xac\x1c\x26\xdf\x2a\x0d\xcf\x5d\xb7\x07\x0e\xac\x26\x2a\x80\x71\xda\x16\xaf\xa7\xc4\x19"[..]);

        ctx.update(b"\x98\x1f\x41\xa8\x3d\x8f\x17\xf7\x1f\xc0\x3f\x91\x5a\x30\xcd\x8a\xc9\x1d\x99\xaa\x1b\x49\xef\x5c\x29\xfb\x88\xc6\x86\x46\xb9\x3a\x58\x8d\xeb\xcd\x67\x47\x4b\x45\x74\x00\xc3\x39\xcc\xa0\x28\x73\x1d\xf0\xb5\x99\x87\x5a\xb8\x0d\xf6\xf1\x8b\x11\xb0\xb1\xc6\x2f\x2a\x07\xb3\xd8\x20\x94\x02");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x62\xae\xa8\x76\x07\x59\xa9\x96\xf4\xd8\x55\xe9\x9b\xcd\x79\xe9\xa5\x7e\xa3\x62\x52\x2d\x9b\x42\xfd\x82\xc1\x2c\x92\x94\xa2\x17"[..]);

        ctx.update(b"\x5c\x58\x9f\xc5\x4f\xef\xc4\xd6\xe2\x24\x9a\x36\x58\x3e\x19\x92\xfc\x6b\x8a\x9c\x07\x0e\x8e\x00\xc4\x5a\x63\x9a\xf2\x20\x63\xe6\x6a\xe5\xcd\xb8\x02\x38\xc8\x2d\xb0\x43\xa5\xe1\xf3\x9f\x65\x62\x6e\x6d\x7b\xe5\xd6\xa2\xd3\x38\x0f\xa2\x12\xf8\x92\x11\x20\x04\x12\xe5\xe4\x31\x5f\xc0\x4e\x40");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x18\xde\xba\x74\xe9\xd9\x3a\xe7\xdf\x93\xc6\xc3\x16\xef\x20\x1b\xf5\xe3\xa6\x61\xe6\x88\x68\xe1\x4d\x4f\x56\x26\x4f\x5d\x85\x8c"[..]);

        ctx.update(b"\x7c\x86\x91\xe7\xb2\x56\x0f\xe8\x7f\xcc\x5e\x28\x77\xf7\xe3\xc8\x4d\x91\x01\xec\xa4\x81\x8f\x63\x22\xa5\x89\x86\xc6\xcf\x05\x62\x7c\x0d\x69\x19\xef\x2e\xdc\x85\x9f\x81\xfa\x1f\x33\xe0\xcc\x1f\x10\xed\xf7\xe5\x2a\x9c\x33\x98\x1a\xf2\xff\x0d\x72\x0c\x94\xea\x4d\x62\x17\x0b\x2a\x4d\x12\x24\xfa");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x5a\x5a\x43\x8b\x57\xc1\xb3\xce\x87\x56\x09\x42\x52\x36\x2a\xfe\xaa\x9f\xc9\x1c\xd4\x5b\x38\x5d\x16\x99\x4e\xc8\xaf\x49\xaa\x6b"[..]);

        ctx.update(b"\x97\x35\x9b\x56\x4b\x2b\xc2\x08\x00\xed\x1e\x51\x51\xb4\xd2\x58\x1a\x04\x27\xce\x95\x39\xd3\x24\xc3\x63\x7c\xfb\x0e\x53\x78\xdc\x2c\xf6\xd7\x29\x46\xe2\xa3\x53\x5a\x2f\x66\x4e\xde\x88\xed\x42\xa6\x81\x4c\x84\x07\x2b\x22\xc4\x3d\xe7\x1e\x88\x0a\x77\xc2\xd9\xa0\x5b\x67\x3b\xc1\x5a\x82\xe3\x25\x5f");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xbe\x54\xf2\xe4\x35\xf7\x60\xd5\xb7\x7c\x0a\xe6\x1e\xf0\xaa\x7f\x5f\x33\x66\xf4\x78\x19\xf3\x50\xdc\x8a\x39\xaf\xf8\xc7\x3a\x8f"[..]);

        ctx.update(b"\xa0\xdf\xae\xcd\x3e\x30\x7c\x5d\xdf\x9a\x93\x60\x3f\x7e\x19\x72\x5a\x77\x92\x18\x73\x49\x04\x52\x5b\x14\x58\x6f\xf0\xce\x04\x25\xe4\xef\xe7\xe1\xc0\x6e\x74\x5c\x28\xed\x13\x6f\x60\x31\xc4\x28\x0f\xd4\x06\x1d\x43\x3e\xf7\x00\xb6\xd1\xbc\x74\x50\x64\x23\x1f\xec\xf3\x87\x01\x5f\x94\xf5\x04\xb6\xad\x8c");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x60\xd8\x0f\x1c\x70\x3d\xad\x5d\xa9\x3d\xb2\x22\xfb\x45\xfb\x7f\xa7\x68\xc8\xaa\x27\x87\xf4\xb8\x1f\x1e\x00\x36\x5b\x8f\x49\xe2"[..]);

        ctx.update(b"\x56\x8d\x66\xd0\x61\x30\x6c\x34\x19\xa1\x92\x8c\xe7\xed\xc8\xe3\x40\x0c\x30\x99\x8f\x09\xbd\xac\x6f\x63\xff\x35\x1e\xb2\x3d\x36\x2e\x8d\xc5\x92\x7e\xac\x80\x5d\x69\x4a\xc9\x56\x3d\xcd\x7f\xb2\xef\xa9\x59\x1c\x0d\x82\x7a\xf9\xf3\x91\x46\xf0\x42\x48\x73\xaa\x8e\x39\x63\xd6\x57\x34\xb1\x71\x3b\xaf\x0a\x44");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x7a\x4f\xe3\x7f\x29\x69\x91\x12\x17\x92\xdd\x7c\x2c\x30\x39\x07\x25\xa1\xee\xbb\xf2\x0b\x76\x6a\x5a\x1c\x3c\x6c\x36\x46\xd9\x96"[..]);

        ctx.update(b"\xd6\x5b\x9f\x88\x1d\x1f\xc7\xf1\x7d\x6d\xd4\x29\xfa\xca\x84\x04\xe6\xce\x60\xfb\xa7\xd8\x9b\x7f\xba\x00\x3c\x8e\xf8\x4d\x80\x83\x18\x29\x79\x32\x76\x11\xfc\x34\x12\x91\xba\x80\xdc\x70\xad\x3b\x2f\x28\xb6\xd2\x9b\x98\x84\x45\xe7\xfd\xb7\xc6\x56\x1f\x45\x82\x2a\xc8\x1d\xbf\x67\x7a\x0b\x27\xd9\x61\xdc\x63\x58");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x51\xcc\x71\xb6\x93\x4a\xfc\xf2\x8f\xa4\x99\x42\xb7\x63\x23\xf3\x6c\xd6\xa0\xae\xcc\x5a\x0e\x49\xc1\x09\x94\xdd\xca\xbd\xbb\x80"[..]);

        ctx.update(b"\x71\x1c\x88\xad\xf1\x3e\x7a\x0e\x69\x46\x52\xf2\xb9\xa3\x97\x54\x3f\x49\x37\xfa\xfb\x4c\xcc\xa7\xf1\xad\x1d\x93\xcf\x74\xe8\x18\xd0\xfe\xdf\xae\xe0\x99\xf0\x19\x01\x4e\xc9\xe1\xed\xfe\x9c\x03\xfd\xb1\x1f\xe6\x49\x2a\xd8\x90\x11\xbf\x97\x1a\x5c\x67\x44\x61\xde\x15\xda\xff\x1f\x44\xb4\x7a\xda\xd3\x08\xba\xa3\x14");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x17\x80\xe5\x2e\x30\x68\x58\x47\x82\x90\xc4\x6b\x04\xd8\x06\x8f\x07\x8a\x7f\x6a\xd8\xe3\x79\x0a\x68\xfc\x40\xdc\xcf\xbd\xad\xc9"[..]);

        ctx.update(b"\xf7\x14\xa2\x7c\xd2\xd1\xbc\x75\x4f\x5e\x49\x72\xab\x94\x0d\x36\x6a\x75\x4e\x02\x9b\x65\x36\x65\x5d\x97\x79\x56\xa2\xc5\x38\x80\x33\x24\x24\xdd\xf5\x97\xe6\x86\x6a\x22\xbf\xca\x7a\xa2\x6b\x7d\x74\xbc\x4c\x92\x50\x14\xc4\xed\x37\xbf\xe3\x72\x45\xfa\x42\x62\x8d\x1c\x2e\xe7\x5d\xc9\x09\xed\xc4\x69\xee\x34\x52\xd8\x94");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xf4\xaf\xa7\x2f\x3e\x48\x9a\xd4\x73\xdc\x24\x7a\xae\x35\x3d\xa9\x9f\xb0\x05\xb4\x90\xe2\xc4\xe1\xf5\xbd\x16\xa9\x97\x32\xb1\x00"[..]);

        ctx.update(b"\xfe\x0c\x32\x80\x42\x2c\x4e\xf6\xc8\x21\x16\xe9\x47\xda\x89\xf3\x44\xd6\xff\x99\x7b\xf1\xae\xc6\x80\x7e\x73\x79\xa6\x95\xd0\xba\x20\xae\x31\xd2\x66\x6f\x73\xbb\xdb\xc3\xa6\xd6\xac\x2c\x12\xdc\xfb\x5a\x79\x17\x3d\xfc\x9c\xd2\xe0\xd6\x00\x0e\x31\x14\xf2\x76\x7e\xde\xc9\x95\x77\x2c\x6b\x47\xda\xdc\x13\x6d\x50\x02\x51\xe5");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x89\x19\x8e\x23\x63\xef\xd4\xe0\xba\x7a\x8a\x45\xf6\x90\xf0\x27\x12\xe6\xf8\x56\x66\x85\x17\xba\xe1\x18\xd1\x1e\x9a\x9d\xc7\xcc"[..]);

        ctx.update(b"\x02\xe2\x38\x46\x1d\x0a\x99\xd4\x9c\x4c\xd1\x6f\x44\x2e\xdf\x68\x2c\x39\xb9\x31\x14\xfc\x3d\x79\xf8\x54\x6a\x99\xe5\xea\xd0\x2f\x0c\xfc\x45\x08\x15\x61\xda\x44\xb5\xc7\x0e\xb4\x83\x40\x41\x87\x07\xfd\x6b\x26\x14\x58\x0d\x5c\x58\x18\x68\xba\x32\xf1\xee\x3a\xc3\x4b\xf6\x22\x48\x45\xb3\x2b\xa7\xf8\x67\xe3\x47\x00\xd4\x50\x25");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xab\xef\x81\xb3\x35\x91\xee\xdc\xac\x0c\xf3\x2f\xb5\xa9\x1c\x93\x1f\x2d\x71\x9c\x37\x80\x14\x09\x13\x35\x52\x17\x0c\xe5\x0d\xbf"[..]);

        ctx.update(b"\xfb\x7c\x8c\xd4\x03\x10\x07\xf8\x15\x9d\x5c\x4c\x61\x20\xde\xe6\x77\x7a\x3a\xce\x0a\x24\x5b\x56\xf3\x1e\x8a\xae\x78\x28\xda\xb3\xcf\x35\xc3\x08\xde\x1d\x0d\x68\x45\x92\xef\x3a\x9e\x55\x79\x66\x03\xa9\x2f\x68\xd1\x09\xf7\xa3\xac\x16\x35\xf7\xc4\xd3\x34\x95\x56\x14\xc8\x12\x75\x34\x31\xbb\x0a\x07\x43\x29\x1a\x0f\xc4\x15\x47\xf3");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x5a\x67\x28\x4d\x39\xe4\xf3\x7c\xaa\x64\xca\x1a\x54\x59\x3c\x35\xf6\xd8\xf3\xa3\xec\x20\xd4\x60\x39\x3a\x39\xf6\xf5\x7c\x44\x86"[..]);

        ctx.update(b"\x6b\x2e\x86\x8c\x7d\x0e\xe1\xc2\x40\xd3\xa6\x7e\x2f\xdf\x36\xe8\xe2\x38\x17\xc0\x26\x44\xa5\x44\x53\xd1\x04\x54\xda\x58\x59\xd4\x1e\x83\x3a\x52\x85\xec\x63\xe8\xce\x28\xaa\x64\xa5\x04\x35\xa7\x74\x0e\xea\x4b\x7d\x58\x27\x89\x26\x78\xb3\x59\x93\xd3\xf5\xda\x7a\x1c\x64\xf5\x33\x17\x3f\x3d\x0f\xa3\x7e\x1a\xeb\xf7\x08\x27\x05\x2c\x26");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xae\xcf\x5d\xab\x6f\xea\x9f\xfd\x1b\xce\x2c\xdf\xee\xc0\xbe\xe9\xd2\x14\xa6\x69\xe8\x30\x6d\x5b\x66\x88\xaf\xa8\x95\x7f\xc9\x1f"[..]);

        ctx.update(b"\xe5\xf3\xba\x00\x0c\x43\xbb\x6a\xca\x4e\x0a\x71\x1a\x75\x91\x2a\x48\x24\x1c\xff\xa5\xb4\xb0\xb1\x7f\x90\x1f\x9e\x50\x97\xd9\x40\x36\xc2\x05\xf7\xa3\x07\xd0\x08\x56\x7d\x05\xe5\x8a\xc0\xdf\xaf\x6d\x97\x1b\xf9\xd3\xd4\x50\xcf\x2c\x7c\x83\xf6\xb3\x28\xf6\x76\xe9\xab\x42\x56\x42\xf5\xa5\xa7\x1e\x38\x9d\xc4\xfa\x49\xb6\xd7\xe8\x48\xa0\x9f");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x18\x2d\x6e\x43\x16\xf4\xbc\x18\xd7\x16\x3b\x1b\x21\x46\x2d\x99\xf9\x9c\x6f\x34\xd2\xc0\x0e\xe7\x71\xce\x54\xfd\x6c\x50\x18\xb9"[..]);

        ctx.update(b"\x93\x9c\x61\xe6\x8a\xf5\xe2\xfd\xb7\x5a\x2e\xeb\xb1\x59\xa8\x5b\x0c\x87\xa1\x26\xce\x22\x70\x16\x22\xf5\xc5\xef\x51\x7c\x3a\xb0\xed\x49\x2b\x16\x50\xa6\xc8\x62\x45\x7c\x68\x5c\x04\x73\x21\x98\x64\x5b\x95\xf8\x4c\xcb\x0e\x72\x6a\x07\xce\x13\x28\x27\xa0\x44\xdc\x76\xb3\x4d\x3f\x19\xa8\x17\x21\xf1\xea\x36\x5b\xc2\x3e\x26\x04\x94\x9b\xd5\xe8");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x12\x10\x57\xb0\xb9\xa6\x27\xbe\x07\xdc\x54\xe7\xd1\xb7\x19\xf0\xa3\xdf\x9d\x20\xd2\x9a\x03\xa3\x8b\x5d\xf0\xa5\x15\x03\xdf\x93"[..]);

        ctx.update(b"\x9e\xad\xaf\x48\x11\xa6\x04\xc6\x5e\xaa\x7b\x1c\x6e\x89\xf2\xc0\xab\x96\xbe\xbe\xc2\x5a\x95\x0b\xa7\x8a\xac\x16\xd9\x37\x1c\xa1\xe7\x45\x8a\xcf\x33\x1e\x07\x7e\xf6\xa7\x35\xd6\x84\x74\xab\x22\xd2\x38\x9b\xdf\x35\x7f\xb2\x13\x6c\x9f\x40\xe1\xe1\xeb\x99\x59\x2c\x2b\xbb\x95\xd9\x49\x31\x01\x6b\x4d\x37\xfa\xa0\x8b\x1e\x9b\xf7\x1b\xf2\xd3\x70\x8a");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xc2\x37\x19\x4b\x90\x2e\x48\xdc\xa5\xbd\x09\x6c\xb5\x15\x62\x07\x9d\x0c\xdc\xcb\x2a\xf8\x08\x81\x97\x67\x6c\x17\xb0\x89\x6b\xe2"[..]);

        ctx.update(b"\x71\xdc\xca\x23\x9d\xce\xd2\xac\x5c\xc4\x9a\x9b\xf9\xea\x69\xa9\x9b\xe2\x2b\xa6\x22\x16\x71\x6b\x52\x4d\xb8\x0f\x33\x7d\xee\x5e\xb7\xe0\x32\x86\x9e\x4a\xdc\x14\x97\xba\xbd\x1f\xa8\x2f\xa8\xc3\xcf\xbd\x30\xd2\xea\xdf\xb4\xc5\xd4\x0f\x99\xf9\xd1\x94\xd7\x18\x2c\x9c\xb7\xd4\x1e\x8a\xdb\xdc\xf2\x91\x7e\x08\x67\x82\xfd\xd7\x56\xe2\x96\x1c\x94\x40\x70");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x37\x7d\x1c\xff\xb6\x26\x73\x58\x10\xb6\x13\xfd\x31\xef\x9b\xbb\x45\x77\xcd\x75\x25\x21\xab\xe3\xa4\x1a\xfa\x92\x1e\x62\x3d\xa0"[..]);

        ctx.update(b"\xea\x13\x0d\x32\x36\xbc\xa7\xdf\xfb\x4b\x9e\x50\xe8\x05\x30\x9a\x50\x3e\x73\x47\x22\x7a\xeb\x9f\x1b\xd1\x5c\x26\x3a\x98\xdd\x65\x75\x3d\x2e\xed\xaa\x73\x4b\x9a\xd8\x8f\x41\x15\x8f\x32\x41\x9c\xa5\x29\xf3\x06\x2b\x91\x0c\x01\x9f\x3f\x23\x9f\x63\x5f\xc1\x11\x6e\x5a\xb7\xb2\x42\xfe\xb4\x47\x1e\xd9\x16\x84\x74\xe5\x01\xd3\x9d\x6b\xae\x52\xcc\x21\x06\x1a");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x85\xc7\xa5\x2d\x53\xf7\xb4\x11\x62\xea\x9f\x1e\xf0\xd0\x7c\x3f\xb8\xf0\xec\x62\x16\x17\xf8\x8c\xb3\x82\x8e\xbe\x53\x88\xab\x3d"[..]);

        ctx.update(b"\x28\xf1\xbe\x11\x56\x79\x2a\xf9\x5c\x6f\x72\xe9\x71\xbf\x1b\x64\xe0\x12\x7b\x76\x53\xff\x1e\x8c\x52\x7f\x69\x89\x07\xa2\x7d\x15\x44\x81\x5e\x38\xc7\x74\x55\x29\xbc\x85\x92\x60\x83\x24\x16\xf2\xb4\x1c\xd0\x1e\x60\xc5\x06\x23\x9a\x7b\xf7\x55\x36\x50\xbf\x70\xd1\xfe\x7a\x2c\x12\x20\xac\x12\x2e\xa1\xe1\x8d\xb2\x74\x90\x44\x7d\x85\x45\xa7\x0b\xf0\xff\xc8\xfa");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xb2\xeb\x37\x62\xa7\x43\xd2\x52\x56\x77\x96\x69\x28\x63\xb5\x56\x36\xcb\x08\x8e\x75\x52\x7e\xfd\x73\x06\xa2\xf6\xe3\xa4\x8a\x85"[..]);

        ctx.update(b"\xc8\x40\x0e\xf0\x9c\x13\xe8\xac\xc8\xa7\x22\x58\xf5\xd1\xd2\x03\x02\xc6\xe4\x3b\x53\x25\x0c\x2f\x6c\x38\xff\x15\xbe\x77\xe3\xca\xc0\x4d\x04\xb8\x42\x1f\xc8\xfd\xff\x8b\xe5\xca\x71\xed\xd1\x08\xe9\x28\x7b\x42\xde\xa3\x38\xbf\x85\x91\x00\xee\xa3\x76\xda\x08\xa0\xe6\x95\xf0\xdc\x90\xb9\x5e\x46\x7c\xbd\x3c\x2a\x91\x7a\x50\x4a\x5a\xe0\x1c\x31\x0a\xe8\x02\xc4\xbd");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x69\x96\x6e\x89\xb7\xbc\x7f\x39\xcd\x85\x79\x1b\x92\x18\x0f\xf3\xfe\xd6\x58\xd8\x24\x0e\x39\x3e\x1e\x6d\x7c\x24\xb8\xd0\xac\x95"[..]);

        ctx.update(b"\xa4\x89\x50\xc9\x61\x43\x8e\x09\xf4\xd0\x54\xac\x66\xa4\x98\xe5\xf1\xa4\xf6\xea\xbf\xde\x9b\x4b\xf5\x77\x61\x82\xf0\xe4\x3b\xcb\xce\x5d\xd4\x36\x31\x8f\x73\xfa\x3f\x92\x22\x0c\xee\x1a\x0f\xf0\x7e\xf1\x32\xd0\x47\xa5\x30\xcb\xb4\x7e\x80\x8f\x90\xb2\xcc\x2a\x80\xdc\x9a\x1d\xd1\xab\x2b\xb2\x74\xd7\xa3\x90\x47\x5a\x6b\x8d\x97\xdc\xd4\xc3\xe2\x6f\xfd\xe6\xe1\x7c\xf6");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x44\xc0\x0c\xf6\x22\xbe\xca\x0f\xad\x08\x53\x9e\xa4\x66\xdc\xbe\x44\x76\xae\xf6\xb2\x77\xc4\x50\xce\x82\x82\xfb\xc9\xa4\x91\x11"[..]);

        ctx.update(b"\xe5\x43\xed\xcf\xf8\xc0\x94\xc0\xb3\x29\xc8\x19\x0b\x31\xc0\x3f\xa8\x6f\x06\xac\xe9\x57\x91\x87\x28\x69\x2d\x78\x3f\xa8\x24\xba\x4a\x4e\x17\x72\xaf\xbe\x2d\x3f\x5c\xba\x70\x12\x50\xd6\x73\x40\x5d\x2c\x38\xd5\x2c\x52\x52\x2c\x81\x89\x47\xbc\xc0\x37\x38\x35\xb1\x98\xc4\xcc\x80\xb0\x29\xd2\x08\x84\xac\x8c\x50\x89\x3c\x3f\x56\x5d\x52\x8a\x0c\xb5\x1b\xf8\xa1\x97\xd9\xd6");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x6d\x52\x60\x38\x4f\x3c\xef\xd3\x75\x8f\xb9\x00\xdc\xba\x37\x30\xd2\xb2\x3c\xee\x03\xd1\x97\xab\xef\xf0\x13\x69\xdc\x73\xc1\x80"[..]);

        ctx.update(b"\x4e\x10\xab\x63\x17\x18\xaa\x5f\x6e\x69\xee\x2c\x7e\x17\x90\x8e\xc8\x2c\xb8\x16\x67\xe5\x08\xf6\x98\x1f\x38\x14\x79\x0c\xfd\x5d\x11\x2a\x30\x5c\x91\x76\x2c\x0b\xd9\xdd\x78\xe9\x3e\xf3\xa6\x4c\x8b\xe7\x7a\xf9\x45\xb7\x4f\xf2\x34\xa0\xb7\x8f\x1e\xd9\x62\xd0\xd6\x80\x41\xf2\x76\xd5\xea\x40\xe8\xa6\x3f\x2c\xab\x0a\x4a\x9e\xd3\x52\x6c\x8c\x52\x3d\xb7\xcb\x77\x6b\x98\x25\xb4");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xd8\x8e\x5f\x3b\x2d\x0a\x69\x8f\xd9\x43\x23\x37\x60\xa3\x00\x0a\x33\x60\xd9\x04\x0e\x73\x74\xb2\x2e\x39\xea\x58\xd8\x68\x10\x2d"[..]);

        ctx.update(b"\x60\x4d\x88\x42\x85\x53\x54\x81\x1c\xd7\x36\xd9\x5c\x7f\x46\xd0\x43\xa1\x94\x04\x8b\x64\xbf\x6c\xda\x22\xc3\xe0\x39\x11\x13\xdc\xc7\x23\xe8\x81\xae\x2a\xd8\xdc\x57\x40\xaa\x6b\xda\x66\x69\xdd\xb9\x6b\xb7\x1a\xcd\x10\x64\x83\x80\x69\x3f\x7b\x3d\x86\x2c\x26\x25\x53\x77\x70\x04\xbd\x68\x52\x83\x16\x18\x51\x9f\xbb\x82\x47\x59\xf4\xdd\x65\xaf\x1b\x2a\x79\xcc\x01\x09\x6d\x7c\x8d");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x8a\x8a\xb6\xcf\x5c\x02\xb9\xae\x8f\x4c\x17\x07\x40\xef\xf1\x59\x2f\x3e\xda\x11\xd3\x42\x0a\xc8\xb4\x21\xd9\x3c\xfb\xb3\x5d\xb8"[..]);

        ctx.update(b"\x62\x81\x80\xe1\x4f\x41\xeb\xdf\xde\x3b\x44\x39\xde\x55\xee\x9c\xd7\x43\xd4\x10\x40\xf3\x45\x7e\xf2\x28\x03\x70\xdd\x65\x96\x19\xfa\x0c\xe6\x95\x80\xc7\x09\x72\x5b\x27\x5a\x6e\xda\x8b\xcb\x82\xa8\x44\x7c\x20\xfd\xf6\x8c\xba\x15\x41\x2f\x83\xe2\xa1\x00\x79\xfe\x93\x99\xa3\xe3\xfa\x61\x97\x5e\xc0\xa6\x40\x41\xc0\xec\xde\x59\xe4\x84\x4e\x9f\x8a\x60\x8c\xb2\x2d\x25\x76\x85\x41\x82");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x8d\x15\x4b\xf6\xf9\xcb\x72\xef\xc0\xd8\xb3\x92\x7a\x8f\x69\x00\x60\xd1\xd4\x8b\xbe\x5c\xc7\x20\x94\xd2\xc8\xb1\x49\xa7\x51\x32"[..]);

        ctx.update(b"\xfc\x15\x0b\x16\x19\xd5\xc3\x44\xd6\x15\xe8\x6f\xca\x1a\x72\x3f\x4e\xeb\x24\xfb\xe2\x1b\x12\xfa\xcd\xe3\x61\x5a\x04\x74\x4e\xf5\x4d\x8a\x71\x91\xa4\x45\x43\x57\xde\x35\xdf\x87\x8c\xb3\x05\x69\x22\x78\x64\x87\x59\x68\x19\x19\xd1\xaf\x73\xc1\xfb\x0f\xf9\x78\x36\x78\xae\xc8\x38\xda\x93\x3d\xb0\x37\x6e\x16\x29\xfc\xca\x3f\x32\x91\x3f\x84\xbc\x2f\xf3\xff\xc3\xf2\x61\xd2\x31\x2f\x59\x1c");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x3f\x62\x6c\x8b\xb2\x0a\x13\x24\x95\xbd\x30\x22\xb3\xfc\xd0\xce\x06\x04\xb9\x1a\x9d\x70\x13\x2d\xab\x40\x99\xf7\x3d\xde\x23\xd5"[..]);

        ctx.update(b"\x6d\xad\xbe\xcd\xd1\x5e\x56\x46\xe3\xf3\x7a\x6f\xe5\xb3\x28\xe0\x61\x13\xcc\xe3\xc8\xcf\x07\x28\x59\x39\xaf\xba\x44\xd1\x17\x32\x10\x17\x90\x2b\x3a\x9d\x2f\xf5\x1f\x60\xd1\x8e\x1b\x58\x5d\xcd\xf3\x4e\x49\xe1\x70\xee\x60\xfa\x4d\x1d\xc2\x46\x54\x8d\x2c\x1f\xc3\x8e\x79\x83\xf4\x27\x69\xc4\x3d\x65\xa2\x80\x16\xf3\xf4\xd4\x79\xeb\xe1\xcd\x8f\xec\x5d\x1f\x88\x6d\xd2\x1a\xca\x50\x67\xd9\x4f");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x90\x98\xea\x34\xc4\x0b\x54\x1b\x15\x3e\x80\xa8\xbd\x92\xda\x19\x43\x2b\x18\xb7\xd3\x29\x76\x0b\x30\x2f\x8a\x54\xc3\x95\xdd\x06"[..]);

        ctx.update(b"\x9c\xc5\xfd\x30\x35\xb7\x2d\xc6\x3b\x8c\x3c\x32\x6f\xd0\x13\x08\x1e\x6b\x87\x16\xf5\x26\xd3\xfe\x17\x6b\x45\x25\x6d\x4c\x37\xcc\x3d\xc8\x41\x7d\xff\x49\xad\xa9\x6c\x70\x2b\x8f\xd7\x15\xc6\x5f\xc0\x8a\x17\xa0\xa7\x20\xb9\xcf\x1e\xed\xfd\x49\x22\xcc\xde\x6b\xab\xa4\x37\xf7\x82\xee\x33\xb9\x53\x71\x05\x6b\x03\x50\xda\xd7\x43\x47\x0c\x3b\x66\x32\x99\xf1\x6f\xcf\xd3\x4f\x6f\xc4\x59\xcd\x0e\xe4");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xb0\xc0\x4f\x24\xbb\x6d\x3d\x4f\xcb\xfd\xf9\x22\x2d\x0e\x88\x6f\x1e\xb6\x0a\x05\x66\xa4\x78\x08\x5f\x76\x23\xa0\x25\xa5\xb9\x81"[..]);

        ctx.update(b"\xf3\xf0\x63\xfb\xcf\x2d\x74\xaa\x5a\x02\xd2\x40\xc9\x62\xed\x7b\xb1\x19\xb3\xa2\x12\xbd\xb4\x15\x94\xe2\x84\x28\x10\x8e\x61\x31\x52\xed\x16\xe0\x1e\x45\x1f\xcf\x70\x2b\x0e\x5a\x08\xf8\x2e\xb1\x26\x77\x65\x2b\x93\xe0\x5f\xde\xe0\x0a\xe8\x6c\xf2\xdc\x9a\x1f\xbf\x05\xb9\x39\x52\xec\x5b\x85\x15\xea\xcc\x32\x4f\xb8\x30\xe1\xec\x23\x6a\xfd\x7d\x07\x3d\x4b\x7f\x7a\xb1\xc2\xe0\x48\xb9\x9c\xbf\xa0\x12");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xf9\x30\xd7\x93\x60\xb5\x81\xb1\xbb\xfd\xea\xc5\x71\x33\xa3\x39\x44\x4f\x5c\x44\x53\x8c\x92\x16\x31\xea\xba\xf0\x58\x27\x7d\x32"[..]);

        ctx.update(b"\x84\x07\x39\xa3\xd6\x99\x2c\x13\xec\x63\xe6\xdb\xf4\x6f\x9d\x68\x75\xb2\xbd\x87\xd8\x87\x8a\x7b\x26\x5c\x07\x4e\x13\xab\x17\x64\x3c\x2d\xe3\x56\xad\x4a\x7b\xfd\xa6\xd3\xc0\xcc\x9f\xf3\x81\x63\x89\x63\xe4\x62\x57\xde\x08\x7b\xbd\xd5\xe8\xcc\x37\x63\x83\x6b\x4e\x83\x3a\x42\x17\x81\x79\x1d\xfc\xae\x99\x01\xbe\x58\x05\xc0\xbb\xf9\x9c\xca\x6d\xaf\x57\x46\x34\xec\x2c\x61\x55\x6f\x32\xe6\x42\x73\x05\x10");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x19\x79\x56\x57\xe0\x8c\xfb\xb2\x47\xa1\x7c\xf2\x09\xa4\x90\x5f\x46\xe4\xdd\xf5\x8e\xea\x47\xfe\xee\x0b\xe9\xbb\x9f\x5c\x46\x0f"[..]);

        ctx.update(b"\x4a\x51\xb4\x93\x93\xab\x4d\x1b\x44\xfb\x6d\xc6\x62\x88\x55\xa3\x4e\x7c\x94\xd1\x3b\x8b\x21\x42\xe5\xd5\xa7\xbf\x81\x0e\x20\x2c\xef\xdc\xa5\x0e\x37\x80\x84\x4a\x33\xb9\x94\x2f\x89\xe5\xc5\xb7\xdd\x6a\xfb\x0a\x44\x54\x1d\x44\xfb\x40\x68\x78\x59\x78\x0a\xf5\x02\x5f\xec\xc8\x5e\x10\xcf\x82\x49\x42\x9a\x3b\x0c\x6f\xf2\xd6\x8c\x35\x0c\x87\xc2\xfc\xbf\x93\x6b\xd9\xde\x57\x01\xb2\xc4\x8c\xe9\xa3\x30\xc9\xee");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x12\x8f\xb4\x11\x4e\x43\xee\xfd\x19\x27\x7c\x70\x8b\xe9\xe6\x87\x3e\x66\xd7\xfd\x59\xc5\x8a\x14\x85\xb7\xb0\x15\xfa\xcf\xa7\x95"[..]);

        ctx.update(b"\xaf\xc3\x09\xe6\xb7\xb7\x4d\xfb\x0d\x36\x8e\x38\x94\x26\x6f\xc4\xa7\x06\xc3\x32\x5e\x21\xf5\x55\x0d\x07\xa6\x56\x0e\x3d\x97\x03\xc1\x34\xca\x6a\xd0\x78\xe4\xa7\xb8\x2a\xd6\xfa\x85\xb0\xbc\x1d\xdc\xab\x05\xd4\x3f\x29\xd5\xc5\x8d\x1d\xa7\x8a\xc8\x0c\x37\x05\x1b\x08\x9f\xf3\x1c\xe2\xc0\xc4\x4e\x9c\xe3\xab\xea\x1d\xa0\xf1\xdf\x28\x00\x8e\x17\x8f\xde\xfa\xfc\xa4\x93\x41\x3b\xf1\xd2\x56\xc7\x29\xd0\xa9\x22\x5e");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x03\xe7\x82\xb0\x1a\x4b\xa1\x0f\x64\x04\x70\xbb\x3c\xae\x48\x7e\xb9\xcb\xba\xab\x8c\x99\x41\x97\x8b\x19\x4f\x6a\x31\x2c\xf7\x9e"[..]);

        ctx.update(b"\xc5\xae\x75\x0f\x22\x30\x64\x20\x92\x39\x7b\x84\xad\x55\x26\xc4\x6a\xe9\x48\x0a\xda\x16\x89\x28\x16\xe0\xf2\xdb\x76\x90\xb7\x51\x03\x56\x53\xea\x2f\x33\xda\x3c\xc4\x16\x8b\x59\x1b\x46\xa5\x54\x8e\xff\x7d\x01\x2f\x60\xcc\xfd\xbb\x85\x4d\xee\xc9\xf0\x88\x0c\x47\x2d\xe8\xe1\x27\xb5\x14\x4c\x56\x14\x7c\xcc\xee\x47\x32\xfb\xac\x68\xfc\x59\xa4\x8d\xa7\x4b\x33\xed\x9e\x64\x36\x44\xbb\xe2\x79\x79\x5c\x7c\x73\x7e\xba");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xf6\x4b\x7a\xb2\x43\xce\x6e\x6c\x04\xb4\x83\x88\x8b\xa8\xa6\x55\x46\x5c\x21\xd9\x5e\xb6\x0c\x7b\x8d\x6e\x56\x6a\x38\x11\xba\xe2"[..]);

        ctx.update(b"\x60\x3e\x13\xf6\x14\x99\xe1\x2e\xc6\xb3\x3b\x68\x84\x7a\x28\x1d\x31\x4f\x54\xdc\x70\x5c\x0f\x3f\xc4\x28\x98\x1f\xf5\x68\x9c\x04\xb5\x19\xfa\xdf\x83\xcb\xc9\xfc\xd0\x40\x9c\x32\x60\x35\x04\x5d\xf4\x80\x57\x0e\x26\x5b\xb0\x80\x94\x00\x37\xce\x40\x76\xa3\x64\x37\xaa\xfd\xb3\x71\xc1\xa6\x2a\xf9\xad\x9b\x61\x4d\xfe\xf8\x97\x08\xfb\xbb\x5e\xbe\xf2\xcb\x95\x28\xcc\x39\x97\x81\xe4\xc5\xb2\x2f\x1a\xa4\xdb\xa6\x23\x80\x9f");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x5f\x76\x96\x2f\xd3\xd3\x73\xe5\xdb\x29\x53\xc0\x82\x3a\x51\xfe\x81\xf8\x74\x45\x0b\xed\xf7\xe4\x68\x76\x39\x4b\x04\xd3\xef\x66"[..]);

        ctx.update(b"\xe0\x31\x15\xcf\xa1\x9e\xfc\xd7\x96\xda\x38\x90\x63\xc4\xbe\x6a\xcc\xe6\x84\xd9\x83\xf8\xed\xfb\x3d\xa6\x88\x7b\x0b\x94\xfb\xb5\xe8\x9e\x3a\x1a\x8e\x64\xfd\xd6\x8f\x06\x70\xb1\xa0\x2c\x2c\x33\x38\x4a\x66\x0c\x5a\x22\x66\xb3\xae\x8a\x3b\x4c\xd7\x6f\xae\xcf\x01\x1a\x74\x67\xb9\xb2\xa8\x18\x02\x02\x78\xa5\xa5\x7d\x1e\xb1\xc8\x7f\x12\x24\xc2\xd6\x7d\xd0\x2e\x81\xf1\x55\x3e\xb7\x58\x41\x53\x2c\x2b\x7c\xca\x8f\xe5\xe4\x18");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xd1\x07\xee\x6e\xe4\xa5\x88\x71\xa3\x3c\x49\x65\x7f\xaa\x25\x73\xe4\x75\xf1\x19\x18\xc4\xa4\xe3\x80\x1d\x0e\x17\xfb\x93\xc6\xe3"[..]);

        ctx.update(b"\x0e\x6c\x1d\x58\xb1\xb9\xd3\xa2\xd3\x99\xaa\xfd\x60\x52\x9e\x07\xd4\x83\xa2\x75\x5b\xb7\xe4\x4c\x37\x3b\x53\x55\x63\x2d\x5f\xca\x76\xd6\xff\x56\xc9\x3a\xf9\x3d\xdc\xec\x5e\xd6\xf6\x27\x53\x42\x0c\x1b\x17\x58\xe4\x85\x42\xdf\x7b\x82\x4b\x00\xa3\xa5\x4d\xfa\xf0\x47\x0b\x18\xd5\x1e\x31\xe1\x0b\x12\xdd\x8e\x32\x4b\x5d\xc1\xbb\x8f\x3b\x73\x05\xcb\x76\x2e\xc6\xef\x13\x7d\xad\xff\xd4\xa2\x46\x67\x48\x86\x1d\x90\x04\xf6\x26\xb0");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x02\xab\x2d\xbb\x02\x94\x43\x54\x79\x90\x51\x24\x7b\x1a\x25\xc1\x9f\x36\x96\xe1\xaf\xcb\x50\x2b\x85\x9e\x83\x79\x8b\x33\xfd\x77"[..]);

        ctx.update(b"\x6d\xb2\xa4\x3a\x22\x9b\x10\xc3\x62\x92\x49\xfc\x51\x36\x46\x8b\x4d\x84\xdf\x7b\x89\xec\x90\xeb\xf7\xaa\x7a\x03\x6c\x53\xaa\x2d\xff\xae\x9e\x81\xb2\xc6\x05\x80\x54\x3d\xc7\x06\xa5\xe3\x45\x7a\xbc\x87\xe2\x48\xa6\x0e\xc2\x91\x50\xc2\xd2\x21\xa6\xec\x08\xa1\xfd\xa4\xec\x0d\xae\xe8\x57\x69\x04\xec\x7a\xb0\x59\xb1\x23\x0e\x7b\xd9\x3c\x4e\x55\xba\x94\x96\xcb\xb1\xe3\x52\xe5\xb8\x08\x6e\x30\x3b\x94\xc8\x61\x28\x8c\xe5\x3c\x46\x6b");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x8c\xc4\xd3\x9b\x2f\x5b\xa0\xbc\x9d\x2e\xe2\xa8\x77\x7c\xf0\x85\x33\xe6\x0c\xc6\x9b\x65\xa7\xb3\x1c\x5c\x21\x21\x19\x3a\xa3\x1e"[..]);

        ctx.update(b"\x31\xd9\x95\xf7\xff\x8b\x6d\xe7\x08\x29\xa8\x33\x6c\x61\x0f\x10\xdf\x2c\x86\x61\x07\xa4\x92\x2b\x25\x15\x18\x49\xf8\x56\x68\x61\xdf\x5a\x79\x16\x3d\x02\x76\x7f\x21\x35\x7a\xd8\x27\x33\x99\x78\x99\x26\x1f\x03\xda\xfb\x1c\xe1\x05\x6f\x20\xef\xd1\x6d\x43\x74\xb8\x97\x68\x56\x58\x23\xc3\x8e\x19\xe8\x99\xd9\x10\xb8\x47\xb0\x23\xf1\x86\x7b\x6e\x4f\xed\x02\xe6\x04\xb8\x24\x3c\x0b\xc7\xcb\x05\xb9\xea\x1f\x17\x95\x5b\xfa\x36\x69\x8c\x9c");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xc9\x9c\x71\x91\xb3\x4c\x9a\xd3\xf9\x41\xd4\xad\x44\x2c\xc8\x65\x20\x5c\xbb\x4c\x2a\x69\x27\xc5\x92\xe8\x31\xcb\xc4\xd3\x6f\xcf"[..]);

        ctx.update(b"\xcb\x0b\x8c\xb7\xde\x62\x1c\x8e\x0a\x0f\xc6\xbe\x2f\xc1\x8d\x0e\x88\x18\xa2\xc2\xdd\x0b\x32\x19\xfa\x87\x83\x1a\x61\x58\x3f\x90\x3c\x4d\x10\x54\x95\x97\x6c\xca\xc9\x73\xb3\xae\x3a\x09\x77\x11\x45\x93\x1a\x9e\x74\xc1\x9f\x22\xf4\x5c\xba\x4c\x49\x2b\x29\xb1\x40\x13\x47\x12\x25\x81\xdf\xe2\x37\x0d\x3e\x03\x59\x57\x8c\xd1\x0a\x35\x5c\x61\x97\x11\x81\x0a\x8f\x8c\x23\x25\x78\x67\x13\x12\xc0\xa4\x5c\x7c\xf7\xe8\x1b\xdd\x3b\x24\x90\x44\xf3");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x6d\x2f\x57\xa7\xe4\x2b\x35\x36\x9c\xf2\xcd\x60\xca\xf9\xe6\x5a\xca\x7d\x9a\xa0\x19\xe6\x82\x4b\xb8\x06\x34\x8f\x1a\xcf\x3c\x7c"[..]);

        ctx.update(b"\x48\xdf\xf7\x8a\xed\x5f\x6e\x82\x30\x54\x92\x4a\x78\xdc\x1b\x8e\x51\xa1\x17\xf1\x61\x01\x81\x52\x9f\x6d\x16\x4e\xbf\x0f\x64\x06\xf0\xb0\x24\x22\xca\xd8\xc9\x16\x82\x37\x59\xa3\x61\x43\x7c\xa1\x74\x23\xd3\xfd\x84\xcc\x8a\xfe\x48\x6a\x31\xcc\xda\x01\xc7\x32\x68\x54\x18\xa3\x2c\x06\x4a\x7b\x9e\xff\xb2\x88\xe8\x11\xec\xc9\x9a\xdb\x2a\x75\x9f\xee\xcc\x3f\x70\x2f\x31\xd9\x87\x7d\xcd\xb7\x17\x93\x7c\x15\xfa\x2f\x16\x3b\xea\x74\x44\x00\xf5\x8c");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x14\xb6\x31\xf0\xf0\x0a\x30\x24\xad\x18\x10\xda\xbf\x02\x71\x1e\x28\x44\x96\x68\xab\xe2\x7f\x69\x38\x09\x42\x26\x89\x68\xd4\xf6"[..]);

        ctx.update(b"\x06\xcc\x9f\xa5\x42\xce\xb3\x5c\x88\xfb\x6a\xb8\x2c\x29\xd5\xdc\xd5\x30\xf8\x07\xd3\xf1\xc3\xbc\xb3\x97\x44\x21\x10\x1d\x1a\xa6\xac\x11\x2d\xe6\xbf\x97\x9c\xd2\x8e\xb0\xf7\x0c\x40\xbc\xaf\x91\xed\x3e\xca\x9b\xf9\xe0\xdb\xc6\xa0\xb7\x32\x71\xd1\xc7\x50\x67\x40\xca\x9e\xbf\xb7\x2d\x5e\x00\xac\x5c\xe1\x89\x19\x3f\xfa\x30\x88\x04\xb4\x2a\x6d\x20\x40\x2b\xb9\x90\x31\xcd\xac\x65\xec\x36\xeb\x7f\x59\xf5\xd2\x99\xdf\x2e\x0b\x86\x90\xf7\x60\xb9\xa0");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x57\x4f\xd8\x2a\x9f\xce\xb8\xf7\xbb\xbf\x24\x4d\x16\xe0\x41\x2c\xbd\xa8\x15\x3b\x72\x08\x46\xc3\x2b\x8f\x10\xfe\x57\x79\xa8\x81"[..]);

        ctx.update(b"\x8d\x93\x62\x7c\x0b\x7c\xbf\x61\xa7\xfe\x70\xe7\x8c\x2c\x8e\xd2\x3b\x13\x44\xb4\xcf\xed\x31\xbd\x85\x98\x0d\xd3\x7b\x46\x90\xe5\xb8\x75\x8f\x7d\x6d\x22\x69\x95\x7a\x39\xa1\xac\x34\x51\xcc\x19\x66\x96\xae\x9e\x96\x06\xa0\x40\x89\xe1\x34\x56\x09\x5a\x1c\xe1\xe5\x93\x48\x1b\x3a\xc8\x4f\x53\xf1\xcb\x10\xf7\x89\xb0\x99\xf3\x16\xc9\x48\x39\x8a\xd5\x2f\xa1\x34\x74\xbd\xf4\x86\xde\x9b\x43\x1b\xd5\xd5\x7e\xf9\xd8\x3a\x42\x13\x9a\x05\xf1\x12\xb2\xbd\x08");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x34\x4e\xc8\x66\x42\xea\xbb\x20\x6b\x2f\xd9\x30\xe4\xc5\xdd\xe7\x8a\xa8\x78\x57\x7d\x6c\x27\x1c\xb0\x06\x9d\x49\x99\x49\x56\x52"[..]);

        ctx.update(b"\xd0\xaf\x48\x4b\x8b\xe6\xb4\x1c\x19\x71\xae\x9d\x90\x65\x0a\x1e\x89\x43\x56\xc9\x19\x1d\x6b\xe3\x03\xfa\x42\x4f\x2b\x7c\x09\x54\x4e\xc0\x76\xa0\xf1\x86\x5c\x8c\x97\x92\x7c\xa1\x37\x52\x9d\x5b\xed\xc0\xdf\x2e\xf0\x8a\x4c\xc7\xc4\x70\xb0\x94\xb1\xee\xaa\x86\x73\x1c\x04\x16\x33\xd2\x40\x86\xb6\x0f\x73\x69\xd5\x9c\x57\x65\x2d\xec\x9b\x38\x17\x47\x7d\xf9\xdb\x28\x9b\xa0\x20\xe3\x06\xc9\xa7\x8a\x99\xb5\x39\x12\x89\x92\xde\xb2\x3c\xfc\x50\x8c\x5f\xc3\xaf");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xb7\xba\x99\x87\x26\x47\x7c\x32\x79\x2e\x9c\x3e\xdd\xc1\xcb\x6f\xeb\x7c\x39\x33\xe4\x9f\x2e\x75\x90\xd8\xce\x7a\x21\x13\xe6\xf8"[..]);

        ctx.update(b"\xb2\x12\xf7\xef\x04\xff\xcd\xcf\x72\xc3\x9a\x63\x09\x48\x6c\x0e\xeb\x39\x0f\xf8\xf2\x18\xd6\xbd\x97\x8b\x97\x66\x12\xf7\xf8\x98\xc3\x50\xe9\x0b\xd1\x30\x72\x3e\x11\x26\xaf\x69\x29\x50\x19\xb4\xf5\x2c\x06\xa6\x29\xab\x74\xe0\x38\x87\x02\x0b\x75\xd7\x3f\x0f\x78\xe1\x27\x85\xc4\x2f\xeb\x70\xa7\xe5\xf1\x27\x61\x51\x1c\x96\x88\xc4\x4d\xa6\xaa\xa0\x2a\xfa\x35\xb3\x1e\xdc\x94\xc3\xa0\x77\x9b\x6a\xb9\x46\x25\x25\xc0\xcc\xfb\xa7\x69\x86\xf8\x73\xfe\x1e\x6b\xa9");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x2f\x26\xb9\x6c\x1f\xa3\xf3\xde\xe7\x28\xf1\x75\x84\xe7\x33\xb4\x18\x98\x21\xc6\x59\xb8\x88\x5a\x5f\xb1\xd1\x2d\x60\xd2\xaa\xa9"[..]);

        ctx.update(b"\x86\x59\x1a\xda\x83\xfb\xa8\x17\x5a\x0f\xe9\x1d\x26\x4e\x7f\x9b\x2d\xf9\x7e\xe4\xc3\x25\x70\xe7\x6b\x57\x9d\x61\x40\x50\x89\x51\x93\x2a\xbd\xad\xd6\xa4\xca\x53\xb8\xbb\x8c\x42\x92\x7a\xac\x0a\x02\x12\x68\x81\xd5\x2d\x97\xb8\x2b\x80\xe7\x2d\xd5\x9f\x6a\x42\x02\x16\x51\xee\x1b\xb5\xf7\xb3\xeb\x2b\x21\xd0\x03\xd7\x84\xb7\x5d\xda\x87\xc1\x3f\x71\x4b\x21\x62\x82\xe8\x17\x54\x74\xfa\x66\x1b\x44\x5d\x07\x1b\xd5\x34\x1f\x3a\x88\x30\x2f\x41\x0d\x0f\x8a\x85\x79\x62");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xe3\xed\xbc\x8c\x42\xce\x5d\x23\x84\xdf\xb2\x4f\xb1\xde\x5d\x47\x98\xb1\xbc\x3c\xc7\x8c\x97\x03\x38\x94\x04\x0d\xfa\x6f\xeb\x6c"[..]);

        ctx.update(b"\x92\xb5\xa8\xe8\x4b\x6a\x2a\xc4\xd5\xb1\xe6\x1d\x63\x80\x4a\xbd\x64\x1d\xd6\x30\x05\x8e\xc6\xd5\xf7\x52\xf1\x35\x72\x4e\xf1\x94\x7a\x0a\x84\xc6\x61\x1d\x32\x44\x8d\xe6\x30\x7f\x7b\x7d\x85\x74\x04\xe9\x6b\x81\xdf\x94\xf8\x77\x68\xfc\xfd\xf0\x9f\xaa\x2f\xe3\x74\x68\x84\x75\x42\xaf\xe0\x12\x99\x5f\xf1\xbd\x40\xb2\x57\xa4\x7a\x73\x09\xf8\x89\x6b\xf4\xfb\x71\x1d\xe5\x5b\xfe\xb3\xa8\xbe\x08\x37\x72\x9e\xf6\x06\x7c\x57\x81\x82\xf1\x7e\xbb\x08\x0a\x75\x4f\x22\x77\x3c");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x80\xed\x0a\x70\x28\x12\x29\x7c\x2a\xa1\xb6\xb4\xb5\x30\xc2\xb5\xed\x17\xec\xfb\xa6\xd5\x17\x91\xcf\x15\x2d\x43\x03\xce\xd2\xe6"[..]);

        ctx.update(b"\xd2\x84\xa0\xa9\xa4\xde\x5d\x4c\x68\xcc\x23\x88\x4c\x95\xad\x76\x19\xaa\x39\xb2\x0a\x2c\xf4\x01\xde\xae\xb3\x36\x2c\x3c\xe3\x56\xf7\x9c\xc3\xfa\x82\xd3\xd1\xf5\x65\xec\x81\x37\xe1\xf4\x35\xf1\x71\x49\x6a\xfa\xa1\x15\x2f\x72\x23\x15\xdc\xa5\x20\x9f\x00\x31\xcc\xe3\x9b\x6c\x3d\x71\x8e\x00\x7d\xfb\x4f\xd8\xde\x5c\xe1\x40\x8d\xda\x04\x47\x6a\xa8\xa9\x68\x17\xaf\xa8\x6a\x4f\x8f\xb5\x85\x7a\xe0\x91\xc6\x7e\xbd\x7d\xb5\xd7\x83\xf4\x34\xea\xd6\x99\xaa\x96\xe5\x6f\x61\x0d");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x65\x4e\xcc\xef\xd0\xa4\xfd\xb2\xac\x0a\xb5\x62\x88\xc6\x43\x99\xb3\x7b\xc4\xd5\x7f\xf4\xa9\xf1\xcc\xe9\x43\x62\xfc\x49\x1b\xda"[..]);

        ctx.update(b"\xf5\x7f\x0f\x87\x95\x38\x5b\x80\x52\x46\xa0\xa2\x57\x3a\xfc\x27\x43\x46\xa9\xec\xcf\x50\xc6\x26\xb0\x45\x5a\x50\xbf\xb0\x96\x68\x57\x8b\x5a\x5a\xfe\x54\xfb\xbd\x48\x64\x44\xbd\xf9\x7d\xba\x58\x6a\xa2\x24\xce\x2e\x2b\x4b\x52\xf4\x18\xff\x06\xaf\xa6\x5a\x26\xf5\x20\x49\x83\xa5\xf8\x47\x34\xcd\x16\x6c\x88\xcb\x70\xa7\x3f\xb2\xdb\x48\xf9\xef\x20\xc1\xee\x2c\x53\xad\xe0\x74\x60\x11\x4e\x98\xe7\xe2\xeb\xd2\x4a\xc8\x4e\xa9\x04\x22\xeb\x14\x3c\x4a\x42\xe2\x99\x1a\x56\x59\x59");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x13\x5e\xc8\xb1\x44\xa6\x67\xdc\xea\xe8\xfa\xdd\x28\x7d\xf8\x1c\x10\xef\x3e\xbe\xf8\x7f\xf2\xfb\x56\xe6\x0a\xe7\x08\xa8\x8f\x3b"[..]);

        ctx.update(b"\x2a\x41\xa5\x2e\x65\x78\x87\x35\x88\xa5\x7f\x11\xf1\xbe\x7c\x7e\xb3\x98\xd0\x1f\x3b\xfd\xec\x2c\x33\xfe\x6b\x65\xa6\x8a\x53\x4a\x65\x40\x97\x8d\xaa\x82\xe0\xc8\xfc\xcb\x8c\x6c\x52\x42\xf7\xf9\x7b\x8f\xfa\x75\xbd\xed\xb2\x17\xbd\x80\x83\x43\x9e\xea\x5c\xbb\x6d\x19\x3c\x13\xbd\x62\xf5\x65\x8e\xd4\x30\x47\x74\xc6\xb1\xfa\xf5\xb3\xdc\xe4\x32\x48\x78\x40\xca\xba\xb4\x15\xfb\x5d\x67\x64\x0a\x73\x9c\xa6\xe5\x41\x4e\x76\x08\x69\x70\x8a\x9d\x73\x31\xe7\xe7\xad\x7d\x55\xe0\x35\xc7");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xa6\xa1\xb8\xa2\x6f\x6f\x44\x0f\x19\xf1\x6d\xce\x1d\x30\x01\x47\x7d\x73\xee\x7f\x6c\x37\x4b\xce\x29\x22\x16\x7b\x81\x97\x0d\x6a"[..]);

        ctx.update(b"\x4d\x11\xaa\x5d\x3c\x6b\x69\x00\xf4\x9f\xf9\x0d\xd8\x15\x74\x45\x72\xbe\x56\x48\xb6\x4b\xde\x63\x8b\x9d\xb7\xa9\x87\x7d\xd7\x45\xfa\x8e\xa8\x0e\x2f\x7f\x65\x5c\xee\x85\xc7\x1a\x45\x09\xe2\x1d\x89\x9e\x49\xb4\x97\x35\x79\x81\x5f\x94\x75\x87\xa4\x04\xad\x83\xfd\x4a\x24\x80\x20\xd9\xd2\xa6\x5f\x46\x48\x53\x73\xfc\x92\x6d\x79\x31\x61\xf6\x3a\x19\x6a\xe0\xaf\x59\x09\x23\xc5\xbe\x2a\x0e\x5d\x2f\x69\xda\x97\xe0\x78\x85\x50\xc9\xc1\xde\xe9\x57\x4d\xdc\x4a\x61\xe5\x33\x27\x5d\x77\x29");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xfc\x51\x59\xf0\xdd\xd6\xd7\x65\xc8\x5f\xcc\x3f\xc3\xac\x1d\xc0\xd3\x17\xd8\xea\x0b\x11\x0e\x96\xac\x9f\x7a\x39\x8d\xc3\x86\xc5"[..]);

        ctx.update(b"\x05\xcd\x99\xbf\xe0\x31\xd1\x23\xca\x70\x61\xd3\xde\x09\x56\xf4\xbb\xf1\x64\xba\xd7\x92\xdb\x88\x17\x13\xd6\x59\x9d\xda\xb5\x5e\xe2\x4f\xce\xe8\x04\xe3\x60\x89\x61\x52\xc8\x76\x64\x24\xf8\x30\x9f\x7a\x24\x64\x1a\x07\xbe\x0f\xeb\x5d\xa5\xe5\x07\x6a\x9a\xf4\x58\x42\xf3\x85\x10\x1f\x93\x43\x3c\xa5\x19\x9f\x9c\x6b\x58\x72\xb2\xb8\x08\xe4\x19\x8a\xba\x8e\x18\xdd\x12\xdb\x77\x29\x30\xb4\x91\x2d\x6f\x5c\xab\xeb\x52\x98\x84\xf4\xbb\x14\x2d\xe5\x5e\x02\x1b\x32\x76\x04\x7b\x22\xb6\x4c\xc5");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x8a\xa0\x77\x42\xe6\xf1\xf4\x7a\xd0\x20\xed\x66\x84\xed\xc8\xdb\xa4\xaf\x36\xb7\x82\x95\x5f\x0f\x97\x2b\xe3\xae\x98\x0a\xea\x0e"[..]);

        ctx.update(b"\x52\x96\x84\x39\x8d\x68\xbd\xc1\x9e\x7a\x00\xce\x32\xcc\x1a\x8c\x13\x15\xb9\x7f\x07\x13\x74\x74\xf6\x1f\x0c\xb8\x4a\x04\xf2\x87\x9b\x11\x09\xc7\x8c\x6d\xac\xf7\xf0\xab\xf3\x62\x32\x9e\x32\x98\xf3\x6f\xc3\x1e\xf4\xec\x06\x65\x37\x23\xa5\xf9\x61\x30\x1d\xfb\x63\x53\x7a\xd1\x59\x46\x61\x1c\xb2\xcd\x54\xea\x92\x8e\x32\x2e\x74\x23\xfd\x6d\x14\x6e\xe0\xb9\x8c\x2c\x71\xe3\xbd\xcd\x33\xed\xf0\x84\x5f\xbe\xbd\x9a\xe4\x19\x2d\x07\xac\xd0\x1b\x43\x21\x35\xe0\x5a\xf0\xd2\x2f\x3f\x0c\x5a\x3d\x62");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xa0\x70\x49\xb6\xeb\xd7\xb3\x55\x47\x9a\x3d\x80\x2f\xda\x43\x6b\x83\xae\x67\x47\xd7\x41\xcf\x96\x26\xf7\xc6\x2f\x47\xcb\xd5\x63"[..]);

        ctx.update(b"\x98\x2f\xb5\xf4\xaf\x49\x8a\x4a\x75\xe3\x3a\x03\x32\x35\xea\x3d\xdb\x70\xd9\xd2\x36\x51\x9f\x88\x3f\xf5\xb3\x88\xcb\xef\x30\x12\x6b\x98\xd9\x6e\x93\xa6\x5a\x26\xfb\x00\xd1\x72\x46\xd1\x8c\xf4\xe2\xdb\x14\xa5\x2f\x0f\x6b\x10\xe3\x5a\x93\xbe\xad\xc1\x4f\xf1\x18\xb0\x2e\x95\xb3\x8f\xc4\x73\x6f\x97\x3b\xa8\x48\xe4\x0b\x55\x27\xcb\x05\x99\x07\x6d\x96\xbc\x57\x8c\x4a\xad\xa0\x9e\x8f\xaf\x68\x20\xbc\x4f\x56\x2d\x51\x99\x97\x4f\x80\x8b\x7f\x95\xed\xca\x74\xe6\xb3\x94\x08\x94\xa7\xf6\x65\x34\xe0");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x09\xc6\x0f\xec\x5a\x08\x9a\x23\xf5\xda\x3e\xd2\x49\x2a\xa2\x1f\xcf\x7a\xa3\x61\x83\x85\x0f\xaf\xc1\x5a\xe8\xc6\x3f\x59\x6d\xb0"[..]);

        ctx.update(b"\xca\x88\x61\x48\x28\xf8\xac\xdb\x5f\xcf\xfa\xb6\xbb\x2f\xb6\x2d\x93\x2b\x78\x08\xe4\xd9\xcc\x31\x39\xa8\x35\xb0\xce\xf4\x71\xd9\xf4\xd8\xff\xc4\xb7\x44\xdf\xfe\xbf\x4f\x99\x7e\x74\xce\x80\xdb\x66\x25\x38\xbc\xeb\x5d\x76\x8f\x0a\x77\x07\x7e\x97\x00\x14\x9e\xa0\xe6\xa4\x6a\x08\x8a\x62\x71\x72\x16\xa1\x4b\x60\x11\x9d\xd1\x9c\x31\x03\x8e\xd8\x70\xb4\x70\x91\x61\xc6\xc3\x39\xc5\xcc\x60\x94\x5a\x58\x22\x63\xf3\xbe\x9a\x40\xcd\x1a\x04\xc9\x21\x94\x79\x00\xf6\xe2\x66\xf2\x39\x0f\x3c\x97\x0f\x7b\x69");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xfe\x2d\x41\x83\xcc\xda\xa8\x16\xb4\x44\x6a\x9b\x6c\x07\xd0\xba\x4b\x42\xac\x74\x35\x99\xdb\x5d\xc4\x82\xb1\x94\x1f\x44\x3c\x71"[..]);

        ctx.update(b"\xab\x6b\x92\xda\xf8\x32\x75\xcb\x9c\x1b\x76\xcf\xb5\x9f\xbc\xc8\xac\x53\x18\x8e\x0b\x69\x80\x91\x8e\x7a\xc0\xc0\x7c\x83\x6c\xa9\x37\x2d\x19\xe1\x12\x51\xcc\xa6\x64\xbb\xb3\xc3\xdb\x2e\x13\xb4\x12\xa9\x82\x0b\x65\xe9\x56\x12\x04\x2f\x5d\xb2\x46\x43\xcf\x93\x40\xb9\x80\x85\x97\x73\x5a\x1f\x92\x67\x0b\xa5\x73\xa2\xfb\x2f\x08\x8d\x81\x08\x7d\x70\x56\x55\x74\x34\x4a\xf7\x57\x6d\x35\xb2\xed\x98\x31\x8e\x2c\xa0\x06\x7d\x4f\xa8\xe6\x3f\x28\x04\x5b\x83\xb6\x88\x7d\x4f\xfa\x06\x68\xa1\x07\x12\xed\x57\x59");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x74\x45\x38\xe1\xae\x1c\xd7\x35\x77\x10\xb5\x6c\x3b\xc6\xf1\xbd\x7a\x85\x64\x11\x8a\x1e\x0f\x9a\xcc\x30\xfc\xf0\xb5\x39\x6e\xef"[..]);

        ctx.update(b"\xbf\xd4\xc7\xc8\xe9\x08\x58\xcc\xf9\xc8\x83\x4a\xbe\xfd\x9c\x18\x46\xca\x4a\x11\x96\x6f\xdd\x13\x9d\x6d\xe2\x4a\x6b\xeb\xf4\xb1\x9f\x58\xd5\xd5\x1e\x52\xbd\xdd\x0b\xc6\xf1\xc7\xf3\x59\x98\xf4\x47\x07\xca\xe7\x10\x0a\xeb\x4a\xde\xfe\x37\x31\x01\x42\x9d\xa3\xfc\xa1\xd1\x57\x37\x32\x9d\xbb\xf4\x7c\x78\x3a\x84\xde\x59\xbf\xbb\x2f\xcd\x75\xa1\xa1\x48\xd2\x6a\xeb\xb8\xd3\xa9\xa7\x60\x89\xc0\xf8\xe4\xd4\x9b\x71\xa0\x6f\x9e\x32\x3e\x2c\xdb\x54\x88\x81\x89\x88\x7a\x44\xb1\xfa\x9c\xb3\x2b\x7c\x8f\xb7\xc9\xe0");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x58\xb1\x78\x43\xbc\x85\x1a\x72\x1c\x5a\x25\x8e\xef\x57\xb3\x85\x4d\x02\x19\x0e\x73\x2d\x9b\x8e\x7a\x9f\x92\x6a\xc4\x09\xc1\x73"[..]);

        ctx.update(b"\xc5\x01\x94\x33\xc2\x85\xda\x2b\xb9\x3f\x11\x9e\x58\xb4\xf3\x6c\xd1\xe4\xd9\x9d\xda\x35\xdb\xf4\xf8\xae\x39\xc7\xfe\x65\xfa\x0e\xd0\x3b\xd2\xb9\x6d\xc6\x49\x47\x2d\x8f\x1a\x94\x47\x7e\xd9\xf2\x95\x92\xd9\x7c\x9c\xd5\x4d\xa7\xc7\x90\xad\x1a\xf3\xbb\x5c\xc0\x30\xb7\x87\x1b\xc6\x40\x50\xdb\x77\x9d\x2c\xaf\x04\x19\x89\x5b\xf3\xb7\xb5\x0b\x8e\x22\xfb\xe6\x2f\xe3\x0f\xe7\xbb\xd6\xac\xe8\x6d\xdf\x7b\x00\xd5\xd9\x37\x0f\x20\xcf\x0f\x97\x99\x6f\x4b\xce\x70\xbb\x33\xf1\xba\x02\x2c\xda\xba\x0f\x25\xd5\x5f\xa0\x31");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xf7\xc9\x2a\x3f\xb7\xf1\x80\x37\x0d\x62\x8b\xe7\x8d\xe8\x74\xd6\x93\xf7\x4c\xcc\x7a\x54\xc7\x41\x63\x42\x58\xd8\xc5\x12\xfd\x7f"[..]);

        ctx.update(b"\x84\xb6\x0c\xb3\x72\x0b\xf2\x97\x48\x48\x3c\xf7\xab\xd0\xd1\xf1\xd9\x38\x04\x59\xdf\xa9\x68\x46\x0c\x86\xe5\xd1\xa5\x4f\x0b\x19\xda\xc6\xa7\x8b\xf9\x50\x94\x60\xe2\x9d\xd4\x66\xbb\x8b\xdf\x04\xe5\x48\x3b\x78\x2e\xb7\x4d\x64\x48\x16\x6f\x89\x7a\xdd\x43\xd2\x95\xe9\x46\x94\x2a\xd9\xa8\x14\xfa\xb9\x5b\x4a\xae\xde\x6a\xe4\xc8\x10\x8c\x8e\xda\xef\xf9\x71\xf5\x8f\x7c\xf9\x65\x66\xc9\xdc\x9b\x68\x12\x58\x6b\x70\xd5\xbc\x78\xe2\xf8\x29\xec\x8e\x17\x9a\x6c\xd8\x1d\x22\x4b\x16\x11\x75\xfd\x3a\x33\xaa\xcf\xb1\x48\x3f");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x88\x14\x63\x0a\x39\xdc\xb9\x97\x92\xcc\x4e\x08\xca\xe5\xdd\x07\x89\x73\xd1\x5c\xd1\x9f\x17\xba\xcf\x04\xde\xda\x9e\x62\xc4\x5f"[..]);

        ctx.update(b"\x14\x36\x5d\x33\x01\x15\x0d\x7c\x5b\xa6\xbb\x8c\x1f\xc2\x6e\x9d\xab\x21\x8f\xc5\xd0\x1c\x9e\xd5\x28\xb7\x24\x82\xaa\xde\xe9\xc2\x7b\xef\x66\x79\x07\x79\x7d\x55\x51\x44\x68\xf6\x87\x91\xf0\x53\xda\xa2\xdf\x59\x8d\x7d\xb7\xd5\x4b\xee\xa4\x93\xbd\xcb\xb0\xc7\x5c\x7b\x36\xad\x84\xb9\x99\x6d\xca\x96\x35\x41\x90\xbd\x96\xd9\xd7\xfb\xe8\xff\x54\xff\xaf\x77\xc5\x5e\xb9\x29\x85\xda\x50\x82\x5e\xe3\xb4\x17\x9f\x5e\xc8\x8b\x6f\xa6\x0b\xb3\x61\xd0\xca\xf9\x49\x34\x94\xfe\x4d\x28\xef\x84\x3f\x0f\x49\x8a\x2a\x93\x31\xb8\x2a");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x9b\x69\x05\x31\xde\xe9\x48\xa9\xc5\x59\xa2\xe0\xef\xab\x2e\xc8\x24\x15\x1a\x91\x75\xf2\x73\x0a\x03\x0b\x74\x8d\x07\xcb\xaa\x7f"[..]);

        ctx.update(b"\x4a\x75\x7d\xb9\x3f\x6d\x4c\x65\x29\x21\x1d\x70\xd5\xf8\x49\x17\x99\xc0\xf7\x3a\xe7\xf2\x4b\xbd\x21\x38\xdb\x2e\xaf\x2c\x63\xa8\x50\x63\xb9\xf7\xad\xaa\x03\xfc\x34\x8f\x27\x53\x23\x24\x83\x34\xe3\xff\xdf\x97\x98\x85\x9f\x9c\xf6\x69\x3d\x29\x56\x6f\xf7\xd5\x09\x76\xc5\x05\xec\xb5\x8e\x54\x3c\x45\x9b\x39\xac\xdf\x4c\xe4\xb5\xe8\x0a\x68\x2e\xaa\x7c\x1f\x1c\xe5\xfe\x4a\xcb\x86\x4f\xf9\x1e\xb6\x89\x2b\x23\x16\x57\x35\xea\x49\x62\x68\x98\xb4\x0c\xee\xb7\x81\x61\xf5\xd0\xea\x4a\x10\x3c\xb4\x04\xd9\x37\xf9\xd1\xdc\x36\x2b");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x1a\xc7\xcc\x7e\x2e\x8e\xa1\x4f\xb1\xb9\x00\x96\xf4\x12\x65\x10\x07\x12\xc5\xdd\x41\x51\x9d\x78\xb2\x78\x6c\xfb\x63\x55\xaf\x72"[..]);

        ctx.update(b"\xda\x11\xc3\x9c\x77\x25\x0f\x62\x64\xdd\xa4\xb0\x96\x34\x1f\xf9\xc4\xcc\x2c\x90\x06\x33\xb2\x0e\xa1\x66\x4b\xf3\x21\x93\xf7\x90\xa9\x23\x11\x24\x88\xf8\x82\x45\x0c\xf3\x34\x81\x9b\xba\xca\x46\xff\xb8\x8e\xff\x02\x65\xaa\x80\x3b\xc7\x9c\xa4\x27\x39\xe4\x34\x7c\x6b\xff\x0b\xb9\xaa\x99\x78\x02\x61\xff\xe4\x2b\xe0\xd3\xb5\x13\x5d\x03\x72\x33\x38\xfb\x27\x76\x84\x1a\x0b\x4b\xc2\x63\x60\xf9\xef\x76\x9b\x34\xc2\xbe\xc5\xed\x2f\xeb\x21\x6e\x2f\xa3\x0f\xa5\xc3\x74\x30\xc0\x36\x0e\xcb\xfb\xa3\xaf\x6f\xb6\xb8\xde\xda\xcb\xb9\x5c");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xc1\x63\xcd\x43\xde\x22\x4a\xc5\xc2\x62\xae\x39\xdb\x74\x6c\xfc\xad\x66\x07\x4e\xba\xec\x4a\x6d\xa2\x3d\x86\xb3\x10\x52\x0f\x21"[..]);

        ctx.update(b"\x33\x41\xca\x02\x0d\x48\x35\x83\x8b\x0d\x6c\x8f\x93\xaa\xae\xbb\x7a\xf6\x07\x30\xd2\x08\xc8\x52\x83\xf6\x36\x9f\x1e\xe2\x7f\xd9\x6d\x38\xf2\x67\x4f\x31\x6e\xf9\xc2\x9c\x1b\x6b\x42\xdd\x59\xec\x52\x36\xf6\x5f\x58\x45\xa4\x01\xad\xce\xaa\x4c\xf5\xbb\xd9\x1c\xac\x61\xc2\x11\x02\x05\x26\x34\xe9\x9f\xae\xdd\x6c\xdd\xdc\xd4\x42\x6b\x42\xb6\xa3\x72\xf2\x9a\x5a\x5f\x35\xf5\x1c\xe5\x80\xbb\x18\x45\xa3\xc7\xcf\xcd\x44\x7d\x26\x9e\x8c\xae\xb9\xb3\x20\xbb\x73\x1f\x53\xfe\x5c\x96\x9a\x65\xb1\x2f\x40\x60\x3a\x68\x5a\xfe\xd8\x6b\xfe\x53");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x6c\x3e\x93\xf2\xb4\x9f\x49\x33\x44\xcc\x3e\xb1\xe9\x45\x4f\x79\x36\x30\x32\xbe\xee\x2f\x7e\xa6\x5b\x3d\x99\x4b\x5c\xae\x43\x8f"[..]);

        ctx.update(b"\x98\x9f\xc4\x95\x94\xaf\xc7\x34\x05\xba\xce\xe4\xdb\xbe\x71\x35\x80\x4f\x80\x03\x68\xde\x39\xe2\xea\x3b\xbe\xc0\x4e\x59\xc6\xc5\x27\x52\x92\x7e\xe3\xaa\x23\x3b\xa0\xd8\xaa\xb5\x41\x02\x40\xf4\xc1\x09\xd7\x70\xc8\xc5\x70\x77\x7c\x92\x8f\xce\x9a\x0b\xec\x9b\xc5\x15\x6c\x82\x1e\x20\x4f\x0f\x14\xa9\xab\x54\x7e\x03\x19\xd3\xe7\x58\xae\x9e\x28\xeb\x2d\xbc\x3d\x9f\x7a\xcf\x51\xbd\x52\xf4\x1b\xf2\x3a\xeb\x6d\x97\xb5\x78\x0a\x35\xba\x08\xb9\x49\x65\x98\x97\x44\xed\xd3\xb1\xd6\xd6\x7a\xd2\x6c\x68\x09\x9a\xf8\x5f\x98\xd0\xf0\xe4\xff\xf9");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xb1\x0a\xde\xb6\xa9\x39\x5a\x48\x78\x89\x31\xd4\x5a\x7b\x4e\x4f\x69\x30\x0a\x76\xd8\xb7\x16\xc4\x0c\x61\x4c\x31\x13\xa0\xf0\x51"[..]);

        ctx.update(b"\xe5\x02\x2f\x4c\x7d\xfe\x2d\xbd\x20\x71\x05\xe2\xf2\x7a\xae\xdd\x5a\x76\x5c\x27\xc0\xbc\x60\xde\x95\x8b\x49\x60\x94\x40\x50\x18\x48\xcc\xf3\x98\xcf\x66\xdf\xe8\xdd\x7d\x13\x1e\x04\xf1\x43\x2f\x32\x82\x7a\x05\x7b\x89\x04\xd2\x18\xe6\x8b\xa3\xb0\x39\x80\x38\xd7\x55\xbd\x13\xd5\xf1\x68\xcf\xa8\xa1\x1a\xb3\x4c\x05\x40\x87\x39\x40\xc2\xa6\x2e\xac\xe3\x55\x2d\xcd\x69\x53\xc6\x83\xfd\xb2\x99\x83\xd4\xe4\x17\x07\x8f\x19\x88\xc5\x60\xc9\x52\x1e\x6f\x8c\x78\x99\x7c\x32\x61\x8f\xc5\x10\xdb\x28\x2a\x98\x5f\x86\x8f\x2d\x97\x3f\x82\x35\x1d\x11");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x32\x93\xa4\xb9\xae\xb8\xa6\x5e\x10\x14\xd3\x84\x75\x00\xff\xc8\x24\x15\x94\xe9\xc4\x56\x4c\xbd\x7c\xe9\x78\xbf\xa5\x07\x67\xfe"[..]);

        ctx.update(b"\xb1\xf6\x07\x65\x09\x93\x84\x32\x14\x5b\xb1\x5d\xbe\x1a\x7b\x2e\x00\x79\x34\xbe\x5f\x75\x39\x08\xb5\x0f\xd2\x43\x33\x45\x59\x70\xa7\x42\x9f\x2f\xfb\xd2\x8b\xd6\xfe\x18\x04\xc4\x68\x83\x11\xf3\x18\xfe\x3f\xcd\x9f\x67\x44\x41\x02\x43\xe1\x15\xbc\xb0\x0d\x7e\x03\x9a\x4f\xee\x4c\x32\x6c\x2d\x11\x9c\x42\xab\xd2\xe8\xf4\x15\x5a\x44\x47\x26\x43\x70\x4c\xc0\xbc\x72\x40\x3b\x8a\x8a\xb0\xfd\x4d\x68\xe0\x4a\x05\x9d\x6e\x5e\xd4\x50\x33\xb9\x06\x32\x6a\xbb\x4e\xb4\x14\x70\x52\x77\x9b\xad\x6a\x03\xb5\x5c\xa5\xbd\x8b\x14\x0e\x13\x1b\xed\x2d\xfa\xda");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xf8\x2d\x96\x02\xb2\x31\xd3\x32\xd9\x02\xcb\x64\x36\xb1\x5a\xef\x89\xac\xc5\x91\xcb\x86\x26\x23\x3c\xed\x20\xc0\xa6\xe8\x0d\x7a"[..]);

        ctx.update(b"\x56\xea\x14\xd7\xfc\xb0\xdb\x74\x8f\xf6\x49\xaa\xa5\xd0\xaf\xdc\x23\x57\x52\x8a\x9a\xad\x60\x76\xd7\x3b\x28\x05\xb5\x3d\x89\xe7\x36\x81\xab\xfa\xd2\x6b\xee\x6c\x0f\x3d\x20\x21\x52\x95\xf3\x54\xf5\x38\xae\x80\x99\x0d\x22\x81\xbe\x6d\xe0\xf6\x91\x9a\xa9\xeb\x04\x8c\x26\xb5\x24\xf4\xd9\x1c\xa8\x7b\x54\xc0\xc5\x4a\xa9\xb5\x4a\xd0\x21\x71\xe8\xbf\x31\xe8\xd1\x58\xa9\xf5\x86\xe9\x2f\xfc\xe9\x94\xec\xce\x9a\x51\x85\xcc\x80\x36\x4d\x50\xa6\xf7\xb9\x48\x49\xa9\x14\x24\x2f\xcb\x73\xf3\x3a\x86\xec\xc8\x3c\x34\x03\x63\x0d\x20\x65\x0d\xdb\x8c\xd9\xc4");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x4b\xea\xe3\x51\x5b\xa3\x5e\xc8\xcb\xd1\xd9\x45\x67\xe2\x2b\x0d\x78\x09\xc4\x66\xab\xfb\xaf\xe9\x61\x03\x49\x59\x7b\xa1\x5b\x45"[..]);
    }
}
