//light.c:

/*
 *      Copyright (C) Philipp 'ph3-der-loewe' Schafft - 2008-2019
 *
 *  This file is part of roard a part of RoarAudio,
 *  a cross-platform sound system for both, home and professional use.
 *  See README for details.
 *
 *  This file is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 3
 *  as published by the Free Software Foundation.
 *
 *  RoarAudio is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 *
 */

#include "roard.h"

#ifndef ROAR_WITHOUT_DCOMP_LIGHT

static uint8_t nextcycle_events[MAX_EVENTS_PER_CYCLE];
static size_t nextcycle_eventsqueuelen;
static int __cb_event_add(struct roar_slfi_inst * inst, void * userdata, uint8_t event);

// declared 'extern'
struct light_state g_light_state;
struct light_mixer g_light_mixer;
// //

static inline void __set_channel(size_t index, uint8_t val) {
 g_light_state.changes[index] |= g_light_state.state[index] ^ val;
 g_light_state.state[index]    =                              val;
}

int light_init  (uint32_t channels) {
 struct roar_stream_server * ss;
 int i;

 g_light_state.channels = 0;

 if ( channels == 0 || channels > ((uint32_t)512UL*(uint32_t)512UL) ) /* unrealstic values */
  return -1;

 if ( (g_light_state.state = roar_mm_malloc(channels)) == NULL ) {
  return -1;
 }

 if ( (g_light_state.changes = roar_mm_malloc(channels)) == NULL ) {
  roar_mm_free(g_light_state.state);
  return -1;
 }

 if ( (g_light_state.output = roar_mm_malloc(channels)) == NULL ) {
  roar_mm_free(g_light_state.state);
  roar_mm_free(g_light_state.changes);
  return -1;
 }

 if ( (g_light_state.outputchanges = roar_mm_malloc(channels)) == NULL ) {
  roar_mm_free(g_light_state.state);
  roar_mm_free(g_light_state.changes);
  roar_mm_free(g_light_state.output);
  return -1;
 }

 g_light_state.channels = channels;
 g_light_state.filter   = NULL;

 if ( (g_light_mixer.stream = add_mixer(ROAR_SUBSYS_LIGHT, _MIXER_NAME("Light Control"), &ss)) == -1 ) {
  roar_mm_free(g_light_state.state);
  return -1;
 }

 ROAR_STREAM(ss)->info.codec = ROAR_CODEC_DMX512;
 ROAR_STREAM(ss)->info.bits  = ROAR_LIGHT_BITS;
 ROAR_STREAM(ss)->info.rate  = ROAR_OUTPUT_CFREQ;

 for (i = 0; i < ROAR_STREAMS_MAX; i++) {
  if ( g_streams[i] != NULL ) {
   if ( streams_get_subsys(i) == ROAR_SUBSYS_LIGHT ) {
    streams_set_mixer_stream(i, g_light_mixer.stream);
   }
  }
 }

 return light_reset();
}

int light_free  (void) {
 if ( g_light_state.filter != NULL ) {
  roar_slfi_unref(g_light_state.filter);
 }

 if ( g_light_state.state != NULL ) {
  roar_mm_free(g_light_state.state);
 }

 if ( g_light_state.changes != NULL ) {
  roar_mm_free(g_light_state.changes);
 }

 if ( g_light_state.output != NULL ) {
  roar_mm_free(g_light_state.output);
 }

 if ( g_light_state.outputchanges != NULL ) {
  roar_mm_free(g_light_state.outputchanges);
 }

 g_light_state.channels = 0;

 return 0;
}

int light_reset (void) {
 if ( g_light_state.channels == 0 )
  return 0;

 if ( g_light_state.state == NULL )
  return -1;

 if ( g_light_state.changes == NULL )
  return -1;

 memset(g_light_state.state,         0, g_light_state.channels);
 memset(g_light_state.changes,       0, g_light_state.channels);
 memset(g_light_state.output,        0, g_light_state.channels);
 memset(g_light_state.outputchanges, 0, g_light_state.channels);
 memset(g_light_state.events,        0, sizeof(g_light_state.events));

 g_light_state.eventsqueuelen = 0;

 return 0;
}

// called at end of mainloop.
int light_reinit(void) {
 if ( g_light_state.changes == NULL )
  return -1;

 memset(g_light_state.changes, 0, g_light_state.channels);
 memset(g_light_state.events,  0, sizeof(g_light_state.events));

 g_light_state.eventsqueuelen = 0;

 if ( nextcycle_eventsqueuelen ) {
  light_dmxevent_add(nextcycle_events, nextcycle_eventsqueuelen);
  nextcycle_eventsqueuelen = 0;
 }

 return 0;
}

// called at mid of mainloop.
int light_update(void) {
 uint8_t * tmp;
 unsigned int c;

 // swap buffers:
 // after that step we have the old values in outputchanges and output is a fresh buffer.
 tmp = g_light_state.outputchanges;
 g_light_state.outputchanges = g_light_state.output;
 g_light_state.output = tmp;

 // copy data for filters:
 memcpy(g_light_state.output, g_light_state.state, g_light_state.channels);

 // TODO: next run filters
 if ( g_light_state.filter != NULL ) {
  if ( roar_slfi_update(g_light_state.filter,
                        g_light_state.output, g_light_state.channels,
                        ((int32_t)1000000LU)/(int32_t)ROAR_OUTPUT_CFREQ,
                        g_light_state.events, g_light_state.eventsqueuelen) == -1 ) {
   ROAR_WARN("light_update(void): Can not run SLFI filter: %s", roar_errorstring);
  }
 }

 // calculate diffrence:
 // after this step we have only teh changes in the outputchanges array.
 for (c = 0; c < g_light_state.channels; c++)
  g_light_state.outputchanges[c] ^= g_light_state.output[c];

 return 0;
}

int light_check_stream  (int id) {
 struct roar_stream        *   s;
 struct roar_stream_server *  ss;
 char buf[512];
 int i;

 if ( g_streams[id] == NULL )
  return -1;

 ROAR_DBG("light_check_stream(id=%i) = ?", id);

 s = ROAR_STREAM(ss = g_streams[id]);

 switch (s->info.codec) {
  case ROAR_CODEC_DMX512:
    if ( stream_vio_s_read(ss, buf, 512) != 512 ) {
     streams_delete(id);
     return -1;
    }

    for (i = 0; i < (g_light_state.channels < 512 ? g_light_state.channels : 512); i++) {
     __set_channel(i, buf[i]);
    }
//    memcpy(g_light_state.state, buf, g_light_state.channels < 512 ? g_light_state.channels : 512);

    s->pos = ROAR_MATH_OVERFLOW_ADD(s->pos, 1);

    return 0;
   break;
  case ROAR_CODEC_ROARDMX:
    ROAR_DBG("light_check_stream(id=%i): Codec: RoarDMX", id);
    if ( cf_light_roardmx_read(id, ss) == -1 ) {
     streams_delete(id); // simply drop the client on error.
     return -1;
    }
   break;
  default:
    streams_delete(id);
    return -1;
 }

 return 0;
}

static inline int light_send_stream_dmx512  (int id, struct roar_stream * s, struct roar_stream_server * ss) {
 size_t chans = g_light_state.channels;
 uint8_t buf[512];
 register uint8_t * bufptr;

 if ( chans > 512 )
  chans = 512;

 if ( chans == 512 ) {
  bufptr = g_light_state.output;
 } else {
  memset(buf, 0, 512);
  memcpy(buf, g_light_state.output, chans);
  bufptr = buf;
 }

 if ( stream_vio_s_write(ss, bufptr, 512) != 512 ) {
  streams_delete(id);
  return -1;
 }

 s->pos = ROAR_MATH_OVERFLOW_ADD(s->pos, 1);

 return 0;
}

int light_send_stream   (int id) {
 struct roar_stream        *   s;
 struct roar_stream_server *  ss;

 if ( g_streams[id] == NULL )
  return -1;

 ROAR_DBG("light_send_stream(id=%i) = ?", id);

 s = ROAR_STREAM(ss = g_streams[id]);

 switch (s->info.codec) {
  case ROAR_CODEC_DMX512:
    return light_send_stream_dmx512(id, s, ss);
   break;
  case ROAR_CODEC_ROARDMX:
    return cf_light_roardmx_write(id, ss);
   break;
  default:
    streams_delete(id);
    return -1;
 }

 return 0;
}

int light_filter_load(int primary, const char * name, int autoload, const struct roar_keyval * para, ssize_t paralen) {
 struct roar_slfi_inst * filter;
 int ret;
 int err;

 if ( primary && g_light_state.filter != NULL ) {
  roar_err_set(ROAR_ERROR_ALREADY);
  return -1;
 } else if ( !primary && g_light_state.filter == NULL ) {
  roar_err_set(ROAR_ERROR_BADSTATE);
  return -1;
 }

 filter = roar_slfi_new(name, autoload, para, paralen);
 if ( filter == NULL )
  return -1;

 if ( primary ) {
  roar_slfi_cb_set_event_add(filter, __cb_event_add, NULL);
  g_light_state.filter = filter;
  return 0;
 }

 ret = roar_slfi_ctl(g_light_state.filter, ROAR_SLFI_CMD_PUSH, filter);
 err = roar_error;

 roar_slfi_unref(filter);

 roar_err_set(err);
 return ret;
}

int light_dmxchannel_get(size_t index) {
 if ( (size_t)g_light_state.channels <= index ) {
  roar_err_set(ROAR_ERROR_NOENT);
  return -1;
 }

 return (int)(unsigned int)(uint8_t)g_light_state.output[index];
}

int light_dmxchannel_set(size_t index, uint8_t val) {
 if ( (size_t)g_light_state.channels <= index ) {
  roar_err_set(ROAR_ERROR_NOENT);
  return -1;
 }

 __set_channel(index, val);
 return 0;
}

ssize_t light_dmxchannel_num(void) {
 if ( g_light_state.state == NULL ) {
  roar_err_set(ROAR_ERROR_BADSTATE);
  return -1;
 }

 return g_light_state.channels;
}

int light_dmxevent_add(const uint8_t * events, size_t len) {
 size_t i;

 if ( events == NULL ) {
  roar_err_set(ROAR_ERROR_FAULT);
  return -1;
 }

 if ( len > (sizeof(g_light_state.events) - g_light_state.eventsqueuelen) ) {
  roar_err_set(ROAR_ERROR_NOSPC);
  return -1;
 }

 for (i = 0; i < len; i++) {
  g_light_state.events[g_light_state.eventsqueuelen++] = events[i];
  roar_notify_core_emit_simple(ROAR_DATA_DMX512_EVENT, -1, g_light_mixer.stream, ROAR_OT_STREAM, events[i], -1, NULL, -1);
 }

 return 0;
}

int light_dmxevent_read(const uint8_t ** events, size_t * len) {
 if ( events == NULL || len == NULL ) {
  roar_err_set(ROAR_ERROR_FAULT);
  return -1;
 }

 *events = g_light_state.events;
 *len    = g_light_state.eventsqueuelen;

 return 0;
}

static int __cb_event_add(struct roar_slfi_inst * inst, void * userdata, uint8_t event) {
 if ( nextcycle_eventsqueuelen == MAX_EVENTS_PER_CYCLE ) {
  roar_err_set(ROAR_ERROR_NOSPC);
  return -1;
 }

 nextcycle_events[nextcycle_eventsqueuelen++] = event;

 return 0;
}

#endif

//ll
