/*--------------------------------------------------------------------------+
$Id: StringUtilsTest.java 29159 2010-07-08 16:45:10Z heineman $
|                                                                          |
| Copyright 2005-2010 Technische Universitaet Muenchen                     |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package edu.tum.cs.commons.string;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;

import junit.framework.Assert;
import junit.framework.TestCase;

/**
 * Tests for <code>StringUtils</code>
 * 
 * 
 * @author Florian Deissenboeck
 * @author $Author: heineman $
 * @version $Rev: 29159 $
 * @levd.rating GREEN Hash: 4EE3FEE49FFB0D74A383E10BEF9F27EA
 */
public class StringUtilsTest extends TestCase {

	/** Test array. */
	private final static String[] TEST_ARRAY_3_ELEMENTS = { "1", "2", "3" };

	/** Test array. */
	private final static String[] TEST_ARRAY_1_ELEMENTS = { "1" };

	/** Test array. */
	private final static String[] TEST_ARRAY_0_ELEMENTS = {};

	/** Tests if <code>null</code> input works correctly */
	public void testConcatNullInput() {
		String concatenation = StringUtils.concat((Object[]) null, "x");
		assertNull(concatenation);
	}

	/** Tests if <code>null</code> separator works correctly */
	public void testConcatNullSeparator() {
		String concatenation = StringUtils.concat(TEST_ARRAY_3_ELEMENTS,
				(String) null);
		assertEquals("123", concatenation);
	}

	/** Tests if concat works with 3 elements */
	public void testConcatWith3Elements() {
		String concatenation = StringUtils.concat(TEST_ARRAY_3_ELEMENTS, ":");
		assertEquals("1:2:3", concatenation);
	}

	/** Tests if concat works with 1 elements */
	public void testConcatWith1Elements() {
		String concatenation = StringUtils.concat(TEST_ARRAY_1_ELEMENTS, ":");
		assertEquals("1", concatenation);
	}

	/** Tests if concat works with 0 elements */
	public void testConcatWith0Elements() {
		String concatenation = StringUtils.concat(TEST_ARRAY_0_ELEMENTS, ":");
		assertEquals("", concatenation);
	}

	/** Test key value method. */
	public void testGetKeyValuePairs() {
		String testString = "a=1, b=2   ,     c=3,  d , e=5";
		HashMap<String, String> result = StringUtils
				.getKeyValuePairs(testString);
		assertEquals("1", result.get("a"));
		assertEquals("2", result.get("b"));
		assertEquals("3", result.get("c"));
		assertTrue(result.containsKey("d"));
		assertNull(result.get("d"));
		assertEquals("5", result.get("e"));
	}

	/** Test fill string method */
	public void testFillString() {
		String actual = StringUtils.fillString(3, '-');
		assertEquals("---", actual);

		actual = StringUtils.fillString(1, '-');
		assertEquals("-", actual);

		actual = StringUtils.fillString(0, '-');
		assertEquals("", actual);
	}

	/** Test center method */
	public void testCenter() {
		String actual = StringUtils.center("Test", 3, '-');
		assertEquals("Test", actual);

		actual = StringUtils.center("Test", 10, '-');
		assertEquals("---Test---", actual);

		actual = StringUtils.center("Test", 12, '-');
		assertEquals("----Test----", actual);

		actual = StringUtils.center("Test", 11, '-');
		assertEquals("---Test----", actual);
	}

	/** Test flushLeft method */
	public void testFlushLeft() {
		String actual = StringUtils.flushLeft("Test", 3, '-');
		assertEquals("Test", actual);

		actual = StringUtils.flushLeft("Test", 6, '-');
		assertEquals("Test--", actual);

		actual = StringUtils.flushLeft("Test", 7, '-');
		assertEquals("Test---", actual);
	}

	/** Test flushRight method */
	public void testFlushRight() {
		String actual = StringUtils.flushRight("Test", 3, '-');
		assertEquals("Test", actual);

		actual = StringUtils.flushRight("Test", 6, '-');
		assertEquals("--Test", actual);

		actual = StringUtils.flushRight("Test", 7, '-');
		assertEquals("---Test", actual);
	}

	/** Test random string method */
	public void testRandomString() {
		String actual = StringUtils.randomString(10);
		assertEquals(10, actual.length());

		actual = StringUtils.randomString(1);
		assertEquals(1, actual.length());

		actual = StringUtils.randomString(0);
		assertEquals(0, actual.length());
	}

	/** Test random string array method */
	public void testRandomStringArray() {
		String[] actual = StringUtils.randomStringArray(5, 10);
		assertEquals(5, actual.length);

		for (String s : actual) {
			assertEquals(10, s.length());
		}

		actual = StringUtils.randomStringArray(1, 10);
		assertEquals(1, actual.length);

		assertEquals(10, actual[0].length());

		actual = StringUtils.randomStringArray(0, 10);
		assertEquals(0, actual.length);
	}

	/**
	 * Tests whether pseudo-random strings generated from same seed value are
	 * equal.
	 */
	public void testGenerateString() {
		String pseudoRandomString1 = StringUtils.generateString(100, 0);
		String pseudoRandomString2 = StringUtils.generateString(100, 0);

		assertNotSame(pseudoRandomString1, pseudoRandomString2);
		assertEquals(pseudoRandomString1, pseudoRandomString2);
	}

	/**
	 * Tests whether two pseudo-random string arrays generated for the same seed
	 * value are equal.
	 */
	public void testGenerateStringArray() {
		String[] array1 = StringUtils.generateStringArray(100, 50, 0);
		String[] array2 = StringUtils.generateStringArray(100, 50, 0);

		assertNotSame(array1, array2);
		assertEquals(array1.length, array2.length);

		// make sure the array doesn't contain the same string at every index
		assertFalse(array1[0].equals(array1[1]));

		for (int i = 0; i < array1.length; i++) {
			assertEquals(array1[i], array2[i]);
		}
	}

	/** Test is empty method */
	public void testIsEmpty() {
		boolean actual = StringUtils.isEmpty("");
		assertTrue(actual);

		actual = StringUtils.isEmpty("   ");
		assertTrue(actual);

		actual = StringUtils.isEmpty("hallo");
		assertFalse(actual);

		actual = StringUtils.isEmpty("  hallo  ");
		assertFalse(actual);

		actual = StringUtils.isEmpty(null);
		assertTrue(actual);
	}

	/** Test map to string method */
	public void testMapToString() {
		HashMap<String, Integer> map = new HashMap<String, Integer>();
		map.put("test1", 1);
		map.put("test2", 2);
		map.put("test3", 3);
		String[] lines = StringUtils.splitLines(StringUtils.toString(map));
		Iterator<String> keyIterator = map.keySet().iterator();
		for (String line : lines) {
			String key = keyIterator.next();
			assertEquals(key + " = " + map.get(key), line);
		}

		lines = StringUtils.splitLines(StringUtils.toString(map, "->"));
		keyIterator = map.keySet().iterator();
		for (String line : lines) {
			String key = keyIterator.next();
			assertEquals("->" + key + " = " + map.get(key), line);
		}
	}

	/** Test split lines method */
	public void testSplitLines() {

		String[] actual = StringUtils.splitLines(StringUtils.concat(
				TEST_ARRAY_3_ELEMENTS, "\n"));
		assertEquals(TEST_ARRAY_3_ELEMENTS, actual);

		actual = StringUtils.splitLines(StringUtils.concat(
				TEST_ARRAY_3_ELEMENTS, "\r"));
		assertEquals(TEST_ARRAY_3_ELEMENTS, actual);

		actual = StringUtils.splitLines(StringUtils.concat(
				TEST_ARRAY_3_ELEMENTS, "\r\n"));
		assertEquals(TEST_ARRAY_3_ELEMENTS, actual);

		actual = StringUtils.splitLines("hallo");
		assertEquals(1, actual.length);
		assertEquals("hallo", actual[0]);

		actual = StringUtils.splitLines(StringUtils.EMPTY_STRING);
		assertEquals(0, actual.length);

		actual = StringUtils.splitLines(null);
		assertEquals(0, actual.length);
	}

	/** Test <code>indexOf()</code> method. */
	public void testIndexOf() {
		String[] testArray = { "test1", "test2", "test3" };

		assertEquals(0, StringUtils.indexOf(testArray, "test1"));
		assertEquals(1, StringUtils.indexOf(testArray, "test2"));
		assertEquals(2, StringUtils.indexOf(testArray, "test3"));
		assertEquals(-1, StringUtils.indexOf(testArray, "testX"));

		String[] testArray2 = { "test1" };

		assertEquals(0, StringUtils.indexOf(testArray2, "test1"));
		assertEquals(-1, StringUtils.indexOf(testArray2, "testX"));
	}

	/** Test method {@link StringUtils#replaceLineBreaks(String)}. */
	public void testReplaceLineBreaks() {
		String testString = "test1\r\ntest2\rtest3\ntest4";

		assertEquals("test1\\ntest2\\ntest3\\ntest4", StringUtils
				.replaceLineBreaks(testString));

		assertEquals("test1xtest2xtest3xtest4", StringUtils.replaceLineBreaks(
				testString, "x"));
	}

	/** Test method {@link StringUtils#getFirstLine(String)} */
	public void testGetFirstLine() {
		assertNull(StringUtils.getFirstLine(null));
		assertEquals("test", StringUtils.getFirstLine("test"));
		assertEquals("test", StringUtils.getFirstLine("test\ntest1"));
		assertEquals("test", StringUtils.getFirstLine("test\rtest1"));
		assertEquals("test", StringUtils.getFirstLine("test\r\ntest1"));

	}

	/** Test method for {@link StringUtils#getFirstParts(String, int, char)} */
	public void testGetFirstParts() {
		String testString = "edu.tum.cs.commons";
		String result = StringUtils.getFirstParts(testString, 2, '.');

		assertEquals(result, "edu.tum");
	}

	/** Test method for {@link StringUtils#getFirstParts(String, int, char)} */
	public void testGetFirstPartsErrors() {
		String testString = "edu.tum.cs.commons";
		String result = StringUtils.getFirstParts(testString, -1, '.');

		assertEquals(result, testString);

		assertEquals(null, StringUtils.getFirstParts(null, 2, '.'));
	}

	/** Test method for {@link StringUtils#removeLastPart(String, char)} */
	public void testRemoveLastPart() {
		String testString = "edu.tum.cs.commons.string.StringUtils";
		String result = StringUtils.removeLastPart(testString, '.');

		assertEquals(result, "edu.tum.cs.commons.string");
	}

	/** Test method for {@link StringUtils#getLastPart(String, char)} */
	public void testGetLastPart() {
		String testString = "edu.tum.cs.commons.string.StringUtils";
		String result = StringUtils.getLastPart(testString, '.');

		assertEquals(result, "StringUtils");
	}

	/** Test for {@link StringUtils#stripPrefix(String, String)}. */
	public void testStripPrefix() {
		assertEquals("hallo", StringUtils.stripPrefix("pref", "hallo"));
		assertEquals("hallo", StringUtils.stripPrefix("pref", "prefhallo"));
		assertEquals(StringUtils.EMPTY_STRING, StringUtils.stripPrefix("pref",
				"pref"));
		assertEquals("pref", StringUtils.stripPrefix("prefx", "pref"));
	}

	/** Test for {@link StringUtils#stripSuffix(String, String)}. */
	public void testStripSuffix() {
		assertEquals("hallo", StringUtils.stripSuffix("suf", "hallo"));
		assertEquals("hallo", StringUtils.stripSuffix("suf", "hallosuf"));
		assertEquals(StringUtils.EMPTY_STRING, StringUtils.stripSuffix("suf",
				"suf"));
		assertEquals("suf", StringUtils.stripSuffix("sufx", "suf"));
	}

	/** Test for {@link StringUtils#stripDigits(String)}. */
	public void testStripDigits() {
		assertEquals("fbar", StringUtils.stripDigits("f00bar"));
		assertEquals("fbar", StringUtils.stripDigits("f00bar2000"));
		assertEquals("foobar", StringUtils.stripDigits("foobar"));
		assertEquals("", StringUtils.stripDigits(""));
	}

	/** Test for {@link StringUtils#obtainStackTrace(Throwable)}. */
	public void testObtainStackTrace() {
		Exception exception = new Exception();
		String stackTrace = StringUtils.obtainStackTrace(exception);
		assertEquals(Exception.class.getName(), StringUtils
				.splitLines(stackTrace)[0]);
		assertTrue(StringUtils.splitLines(stackTrace)[1].contains(getClass()
				.getName()));
	}

	/** Test for {@link StringUtils#startsWithOneOf(String, String...)}. */
	public void testStartsWithOneOf() {
		String string = "testString";
		assertTrue(StringUtils.startsWithOneOf(string, "test"));
		assertTrue(StringUtils.startsWithOneOf(string, "z", "test", "x"));
		assertFalse(StringUtils.startsWithOneOf(string, "z", "y", "x"));
		assertFalse(StringUtils.startsWithOneOf(string));
	}

	/** Test for {@link StringUtils#endsWithOneOf(String, String...)}. */
	public void testEndsWithOneOf() {
		String string = "testString";
		assertTrue(StringUtils.endsWithOneOf(string, "String"));
		assertTrue(StringUtils.endsWithOneOf(string, "z", "String", "x"));
		assertFalse(StringUtils.endsWithOneOf(string, "z", "y", "x"));
		assertFalse(StringUtils.endsWithOneOf(string));
	}

	/** Test for {@link StringUtils#prefixLines(String, String, boolean)}. */
	public void testPrefixLines() {
		String string = "this\nis\na\ntest";
		assertEquals(StringUtils.normalizeLineBreaks("xthis\nxis\nxa\nxtest"),
				StringUtils.prefixLines(string, "x", true));
		assertEquals(StringUtils.normalizeLineBreaks("this\nxis\nxa\nxtest"),
				StringUtils.prefixLines(string, "x", false));
		assertEquals(StringUtils.normalizeLineBreaks("xthis"), StringUtils
				.prefixLines("this", "x", true));
		assertEquals(StringUtils.normalizeLineBreaks("this"), StringUtils
				.prefixLines("this", "x", false));
		assertEquals(StringUtils.EMPTY_STRING, StringUtils.prefixLines(
				StringUtils.EMPTY_STRING, "x", false));
	}

	/** Check if two string arrays are equal */
	private void assertEquals(String[] array1, String[] array2) {
		assertEquals(array1.length, array2.length);
		for (int i = 0; i < array1.length; i++) {
			assertEquals(array1[i], array2[i]);
		}
	}

	/** Test {@link StringUtils#capitalize(String)} */
	public void testCapitalize() {
		assertEquals("Test", StringUtils.capitalize("test"));
		assertEquals("Test", StringUtils.capitalize("Test"));
		assertEquals("TeST", StringUtils.capitalize("teST"));
		assertEquals("T", StringUtils.capitalize("t"));
		assertEquals("", StringUtils.capitalize(""));
	}

	/** Tests line wrapping. */
	public void testWrapLongLines() {
		testWrapLongLinesWith("This is a text which should be wrapped neatly",
				20);
		testWrapLongLinesWith("This is a text which should be wrapped neatly",
				5);
	}

	/** Test for {@link StringUtils#longestCommonPrefix(Iterable)}. */
	public void testLongestCommonPrefixListOfString() {
		List<String> strings = new ArrayList<String>();
		String prefix = "C:/ConQAT/";
		strings.add(prefix + "a");
		strings.add(prefix + "b");
		strings.add(prefix + "c");

		String result = StringUtils.longestCommonPrefix(strings);
		Assert.assertEquals(prefix, result);

		strings = new ArrayList<String>();
		strings.add("xyz");
		strings.add("abc");
		Assert.assertEquals(StringUtils.EMPTY_STRING, StringUtils
				.longestCommonPrefix(strings));
	}

	/** Tests line wrapping with given parameters. */
	private void testWrapLongLinesWith(String testString, int length) {
		String result = StringUtils.wrapLongLines(testString, length);
		assertEquals("String must be same after normalization", testString
				.replaceAll("\\s+", " "), result.replaceAll("\\s+", " "));
		for (String line : StringUtils.splitLines(result)) {
			if (line.length() > length && line.contains(" ")) {
				fail("Too long lines may not contain multiple words: " + line);
			}
		}
	}

	/** Text pattern with spaces. */
	private static final String SPACE_TEXT = "A B  C   D";

	/** Text pattern with tabs. */
	private static final String TAB_TEXT = "A\tB\tC\tD";

	/** Text pattern with line feeds/carriage returns. */
	private static final String NEW_LINE_TEXT = "A\nB\n\nC\n\n\nD\n";

	/** Tests task tags in single line comments */
	private void assertWhiteSpaceRemoved(String text, String expectation,
			String errorMessage) {
		text = StringUtils.removeWhitespace(text);
		assertEquals(errorMessage, text, expectation);
	}

	/** Tests space removal. */
	public void testSpace() {
		assertWhiteSpaceRemoved(SPACE_TEXT, "ABCD", "Invalid space removal!");
	}

	/** Tests tab removal. */
	public void testTab() {
		assertWhiteSpaceRemoved(TAB_TEXT, "ABCD", "Invalid tab removal!");
	}

	/** Tests new line removal. */
	public void testNewLine() {
		assertWhiteSpaceRemoved(NEW_LINE_TEXT, "ABCD", "Invalid CR removal!");
	}

	/** Tests space and tab removal. */
	public void testSpaceAndTab() {
		assertWhiteSpaceRemoved(SPACE_TEXT + TAB_TEXT, "ABCDABCD",
				"Invalid space or tab removal!");
	}

	/** Tests space and new line removal. */
	public void testSpaceAndNewLine() {
		assertWhiteSpaceRemoved(SPACE_TEXT + NEW_LINE_TEXT, "ABCDABCD",
				"Invalid space or CR removal!");
	}

	/** Tests tab and new line removal. */
	public void testTabAndNewLine() {
		assertWhiteSpaceRemoved(TAB_TEXT + NEW_LINE_TEXT, "ABCDABCD",
				"Invalid tab or CR removal!");
	}

	/** Tests space, tab and new line removal. */
	public void testSpaceTabAndNewLine() {
		assertWhiteSpaceRemoved(SPACE_TEXT + TAB_TEXT + NEW_LINE_TEXT,
				"ABCDABCDABCD", "Invaid space, tab or CR removal!");
	}

	/** Test for regex escaping */
	public void testEscapeRegexReplacementString() {
		assertEquals("a\\\\b\\$c", StringUtils
				.escapeRegexReplacementString("a\\b$c"));
	}
}