/*--------------------------------------------------------------------------+
$Id: ClassTypeTest.java 26283 2010-02-18 11:18:57Z juergens $
|                                                                          |
| Copyright 2005-2010 Technische Universitaet Muenchen                     |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package edu.tum.cs.commons.reflect;

import java.io.PrintWriter;
import java.io.Serializable;
import java.io.Writer;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.SortedMap;

import junit.framework.TestCase;

/**
 * Tests for <code>ClassType</code>. We are mostly interested in the
 * {@link ClassType#merge(ClassType)} and the
 * {@link ClassType#isAssignableFrom(ClassType)} methods, as all others are
 * either trivial or tested implicitly by testing the other two.
 * 
 * @author Benjamin Hummel
 * @author $Author: juergens $
 * @version $Rev: 26283 $
 * @levd.rating GREEN Hash: EE88CDE075D8F48AD7FFE5BAB1617E3D
 */
public class ClassTypeTest extends TestCase {

	/** Test whether the base class is manager correctly. */
	public void testBaseClass() throws TypesNotMergableException {
		assertEquals(Object.class, new ClassType().getBaseClass());

		assertEquals(Integer.class, new ClassType(Integer.class).getBaseClass());
		assertEquals(Integer.class, new ClassType(int.class).getBaseClass());

		assertEquals(Integer.class, new ClassType(Integer.class,
				Serializable.class).getBaseClass());
		assertEquals(Integer.class, new ClassType(Integer.class, Number.class)
				.getBaseClass());
	}

	/** Test whether the interfaces are managed and normalized correctly. */
	public void testInterfaces() throws TypesNotMergableException {
		checkInterfaces(new ClassType());
		checkInterfaces(new ClassType(Integer.class));

		checkInterfaces(new ClassType(Serializable.class), Serializable.class);

		// as Integer is serializable, this should be normalized away
		checkInterfaces(new ClassType(Serializable.class, Integer.class));

		checkInterfaces(new ClassType(List.class, Collection.class,
				Iterable.class), List.class);
		checkInterfaces(new ClassType(List.class, Collection.class,
				ArrayList.class, Iterable.class));

		checkInterfaces(new ClassType(Map.class, List.class,
				Serializable.class, Collection.class, Iterable.class,
				SortedMap.class), List.class, Serializable.class,
				SortedMap.class);
	}

	/** Tests the merging of class types. */
	public void testMerge() throws TypesNotMergableException {
		ClassType ct1 = new ClassType(Writer.class, List.class);
		ClassType ct2 = new ClassType(PrintWriter.class, Collection.class);

		assertEquals(new ClassType(PrintWriter.class, List.class), ct1
				.merge(ct2));
		assertEquals(new ClassType(PrintWriter.class, List.class), ct2
				.merge(ct1));

		try {
			new ClassType(Integer.class).merge(new ClassType(String.class));
			assertTrue("Expected exception!", false);
		} catch (TypesNotMergableException e) {
			// This is expected
		}
	}

	/** Tests the {@link ClassType#isAssignableFrom(ClassType)} method. */
	public void testAssignability() throws TypesNotMergableException {
		assertTrue(new ClassType(Number.class).isAssignableFrom(new ClassType(
				Integer.class)));
		assertFalse(new ClassType(Integer.class)
				.isAssignableFrom(new ClassType(Number.class)));

		assertTrue(new ClassType(Number.class).isAssignableFrom(new ClassType(
				Integer.class, List.class)));
		assertFalse(new ClassType(Number.class, List.class)
				.isAssignableFrom(new ClassType(Integer.class)));

		assertTrue(new ClassType(Number.class, Collection.class)
				.isAssignableFrom(new ClassType(Integer.class, List.class)));

		assertTrue(new ClassType(Iterable.class, Collection.class)
				.isAssignableFrom(new ClassType(Integer.class, List.class)));
		assertTrue(new ClassType(Iterable.class, Collection.class)
				.isAssignableFrom(new ClassType(ArrayList.class)));
		assertFalse(new ClassType(List.class).isAssignableFrom(new ClassType(
				Iterable.class, Collection.class)));
	}

	/**
	 * Tests whether the given class type's interface list consists of exactly
	 * the provided interface classes.
	 */
	private void checkInterfaces(ClassType classType,
			Class<?>... expectedInterfaces) {
		Collection<Class<?>> observedInterfaces = classType.getInterfaces();
		assertEquals(expectedInterfaces.length, observedInterfaces.size());
		for (Class<?> c : expectedInterfaces) {
			assertTrue("Should include " + c.getName(), observedInterfaces
					.contains(c));
		}
	}
}