#ifndef LAUNCHTOOL_OUTPUT_H
#define LAUNCHTOOL_OUTPUT_H

#include <stdio.h>
#include <string>
#include <Logger.h>

// Output and logging system
class OutputMethod
{
public:
	virtual void out(const std::string& str) throw () = 0;
};

class StdoutOutput : public OutputMethod
{
public:
	virtual void out(const std::string& str) throw ();
};

class StderrOutput : public OutputMethod
{
public:
	virtual void out(const std::string& str) throw ();
};

class FileOutput : public OutputMethod
{
protected:
	FILE* oh;

public:
	FileOutput(const std::string& file) throw (FileException);
	virtual ~FileOutput();
	
	virtual void out(const std::string& str) throw ();
};

class SyslogOutput : public OutputMethod
{
protected:
	std::string _identity;
	int _facility;
	int _prio;
	
	void open() throw ();

public:
	SyslogOutput(const std::string& identity, const std::string& facility,
			const std::string& level) throw ();
	virtual ~SyslogOutput();

	virtual void out(const std::string& str) throw ();
};

// Singleton output class
class Output
{
protected:
	static OutputMethod* _lauout;
	static OutputMethod* _lauerr;
	static OutputMethod* _cldout;
	static OutputMethod* _clderr;

public:
	static void set_lauout(OutputMethod* m) throw ()
	{
		if (_lauout) delete _lauout;
		_lauout = m;
	}
	static void set_lauerr(OutputMethod* m) throw ()
	{
		if (_lauerr) delete _lauerr;
		_lauerr = m;
	}
	static void set_cldout(OutputMethod* m) throw ()
	{
		if (_cldout) delete _cldout;
		_cldout = m;
	}
	static void set_clderr(OutputMethod* m) throw ()
	{
		if (_clderr) delete _clderr;
		_clderr = m;
	}

	static void lauout(const std::string& str) throw () { if (_lauout) _lauout->out(str); }
	static void lauerr(const std::string& str) throw () { if (_lauerr) _lauerr->out(str); }
	static void cldout(const std::string& str) throw () { if (_cldout) _cldout->out(str); }
	static void clderr(const std::string& str) throw () { if (_clderr) _clderr->out(str); }

	static OutputMethod* getMethod(const std::string& str)
		throw (ConsistencyCheckException);
};

class LaunchtoolMethod : public Log::Method
{
public:
	virtual void output(Log::LogLevel level, const char* tag, const char* func,
			const std::string& msg) throw ()
	{
		switch (level)
		{
			case Log::DEBUG:
			case Log::INFO:
			case Log::UNUSUAL:
				Output::lauout(msg);
				break;
			default:
				Output::lauerr(msg);
				break;
		}
	}
};

class LaunchtoolLogEnvironment : public Log::Filter
{
protected:
	static const int OUT_NONE = -1;
	static const int OUT_LAUNCHTOOL = 0;
	bool can_log_info;
	bool can_log_debug;
	Log::Method* launchtoolMethod;

	virtual int filter(Log::LogLevel level, const char* tag) const throw ()
	{
		if (level > Log::INFO || (level == Log::INFO && can_log_info) || can_log_debug)
			return OUT_LAUNCHTOOL;
		else
			return OUT_NONE;
	}

public:
	LaunchtoolLogEnvironment() throw () : can_log_info(true), can_log_debug(true)
	{
		launchtoolMethod = new LaunchtoolMethod();
		Log::Logger::instance()->registerMethod(OUT_LAUNCHTOOL, launchtoolMethod);
		Log::Logger::instance()->setFilter(*this);
	}
	virtual ~LaunchtoolLogEnvironment() throw ()
	{
		delete launchtoolMethod;
	}

	void setLogInfo(bool val) throw () { can_log_info = val; }
	void setLogDebug(bool val) throw () { can_log_debug = val; }
};

// vim:set ts=4 sw=4:
#endif
