/* Simple telinit client
 *
 * Copyright (c) 2008-2010  Claudio Matsuoka <cmatsuoka@gmail.com>
 * Copyright (c) 2008-2015  Joachim Nilsson <troglobit@gmail.com>
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#include "config.h"		/* Generated by configure script */

#include <ctype.h>
#include <getopt.h>
#include <stdio.h>
#include <lite/lite.h>

#include "finit.h"
#include "util.h"

static int do_send(struct init_request *rq)
{
	static int fd = -1;
	int result = 255;
	ssize_t len;

	if (fd == -1) {
		if (!fexist(FINIT_FIFO)) {
			fprintf(stderr, "%s does not support %s!\n", prognm, FINIT_FIFO);
			return 1;
		}

		fd = open(FINIT_FIFO, O_RDWR);
		if (-1 == fd) {
			perror("Failed opening " FINIT_FIFO);
			return 1;
		}
	}

	len = write(fd, rq, sizeof(*rq));
	if (sizeof(*rq) == len)
		result = 0;

	return result;
}

static int do_reload(void)
{
	struct init_request rq = {
		.magic    = INIT_MAGIC,
		.cmd      = INIT_CMD_RELOAD,
	};

	return do_send(&rq);
}

static int set_env(int runlevel)
{
	struct init_request rq = {
		.magic    = INIT_MAGIC,
		.cmd      = INIT_CMD_SETENV,
	};

	if (runlevel == 0) {
		strlcpy(rq.data, "INIT_HALT=POWERDOWN", sizeof(rq.data));
		return do_send(&rq);
	}

	return 0;
}

static int set_runlevel(int runlevel)
{
	struct init_request rq = {
		.magic    = INIT_MAGIC,
		.cmd      = INIT_CMD_RUNLVL,
		.runlevel = runlevel,
	};

	return set_env(runlevel) || do_send(&rq);
}

/* telinit q | telinit <0-9> */
static int usage(int rc)
{
	fprintf(stderr, "Usage: %s [OPTIONS] [q | Q | 0-9]\n\n"
		"Options:\n"
		"  -h, --help      This help text\n"
		"  -V, --version   Show Finit version\n\n"
		"Commands:\n"
		"  0               Power-off the system, same as initctl poweroff\n"
		"  6               Reboot the system, same as initctl reboot\n"
		"  2, 3, 4, 5      Change runlevel. Starts services in new runlevel, stops any\n"
		"                  services in prev. runlevel that are not allowed in new.\n"
		"  q, Q            Reload *.conf in /etc/finit.d/, same as initctl reload or\n"
		"                  sending SIGHUP to PID 1\n"
		"  1, s, S         Enter system rescue mode, runlevel 1\n"
		"\n", prognm);

	return rc;
}

int client(int argc, char *argv[])
{
	int c;
	struct option long_options[] = {
		{"help",    0, NULL, 'h'},
		{"version", 0, NULL, 'v'},
		{NULL, 0, NULL, 0}
	};

	progname(argv[0]);
	while ((c = getopt_long(argc, argv, "h?v", long_options, NULL)) != EOF) {
		switch(c) {
		case 'v':
			return puts("v" VERSION) == EOF;

		case 'h':
		case '?':
			return usage(0);
		}
	}

	if (optind < argc) {
		int req = (int)argv[optind][0];

		if (isdigit(req))
			return set_runlevel(req);

		if (req == 'q' || req == 'Q')
			return do_reload();

		if (req == 's' || req == 'S')
			return set_runlevel(1);
	}

	return usage(1);
}

/**
 * Local Variables:
 *  indent-tabs-mode: t
 *  c-file-style: "linux"
 * End:
 */
